<?php

declare(strict_types=1);

namespace Drupal\eca_gmail\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Search for messages in Gmail.
 */
#[Action(
  id: 'eca_gmail_search_messages',
  label: new TranslatableMarkup('Gmail: Search Messages'),
  category: new TranslatableMarkup('Gmail'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Advanced email search using Gmail\'s query syntax with support for filtering by sender, date, labels, keywords, and attachments.'),
  version_introduced: '1.0.0',
)]
final class SearchMessages extends GmailActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'search_query' => '',
      'max_results' => '10',
      'page_token' => '',
      'label_ids' => '',
      'search_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['search_query'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search Query'),
      '#description' => $this->t('Gmail search query using advanced syntax. Examples: "from:user@example.com", "has:attachment", "after:2024/01/01", "in:inbox label:important". Leave empty to list all messages. Supports token replacement.'),
      '#default_value' => $this->configuration['search_query'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['max_results'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Results'),
      '#description' => $this->t('Maximum number of messages to return (1-500). Default: 10.'),
      '#default_value' => $this->configuration['max_results'],
      '#min' => 1,
      '#max' => 500,
    ];

    $form['page_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Page Token'),
      '#description' => $this->t('Token for pagination. Use next_page_token from previous search results to get more messages. Supports token replacement.'),
      '#default_value' => $this->configuration['page_token'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['label_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label IDs'),
      '#description' => $this->t('Optional comma-separated list of label IDs to filter by (e.g., "INBOX,STARRED"). Supports token replacement.'),
      '#default_value' => $this->configuration['label_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['search_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search Results Token Name'),
      '#description' => $this->t('Optional token name to store search results. Accessible as [token_name:messages:0:id], [token_name:total_estimated], [token_name:next_page_token], etc.'),
      '#default_value' => $this->configuration['search_token_name'],
    ];

    $form['scope_warning'] = [
      '#type' => 'item',
      '#title' => $this->t('Important: Gmail API Scope Configuration'),
      '#plain_text' => $this->t('To use search queries (the Search Query field), your Google API Client must NOT include the Gmail Metadata scope. If you have "https://www.googleapis.com/auth/gmail.metadata" checked in your Google API Client configuration, search queries will fail with a "Metadata scope does not support q parameter" error.'),
    ];

    $form['examples_list'] = [
      '#type' => 'item',
      '#title' => $this->t('Query Examples'),
      '#markup' => $this->t('
        <ul>
          <li><strong>from:user@example.com</strong> - Messages from specific sender</li>
          <li><strong>to:me</strong> - Messages sent to you</li>
          <li><strong>subject:"urgent"</strong> - Messages with "urgent" in subject</li>
          <li><strong>has:attachment</strong> - Messages with attachments</li>
          <li><strong>after:2024/01/01</strong> - Messages after specific date</li>
          <li><strong>before:2024/12/31</strong> - Messages before specific date</li>
          <li><strong>in:inbox</strong> - Messages in inbox</li>
          <li><strong>is:unread</strong> - Unread messages</li>
          <li><strong>label:important</strong> - Messages with "important" label</li>
          <li><strong>filename:pdf</strong> - Messages with PDF attachments</li>
        </ul>
      '),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $search_query = $this->tokenService->replacePlain($this->configuration['search_query']);
    $max_results = (int) $this->configuration['max_results'];
    $page_token = $this->tokenService->replacePlain($this->configuration['page_token'], [], ['clear' => TRUE]);
    $label_ids = $this->tokenService->replacePlain($this->configuration['label_ids']);
    $search_token_name = $this->configuration['search_token_name'];

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('search messages');
    if (!$auth_type || !$client_id) {
      return;
    }

    // Build search parameters
    $search_params = [
      'max_results' => min($max_results, 500),
    ];

    if (!empty($search_query)) {
      $search_params['query'] = $search_query;
    }

    if (!empty($page_token)) {
      $search_params['page_token'] = $page_token;
    }

    if (!empty($label_ids)) {
      $search_params['label_ids'] = $this->parseLabelIds($label_ids);
    }

    // Search for messages
    $search_results = $this->gmailService->searchMessages($auth_type, $client_id, $search_params);
    if (!$search_results) {
      $this->logger->error('Gmail search messages action: Failed to search messages using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store search results in token if requested
    if (!empty($search_token_name)) {
      $this->tokenService->addTokenData($search_token_name, $search_results);
    }
  }

}
