<?php

declare(strict_types=1);

namespace Drupal\eca_gmail\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Send an existing Gmail draft message.
 */
#[Action(
  id: 'eca_gmail_send_draft',
  label: new TranslatableMarkup('Gmail: Send Draft'),
  category: new TranslatableMarkup('Gmail'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Send an existing Gmail draft message by its draft ID, useful for workflow approval processes and scheduled sending.'),
  version_introduced: '1.0.0',
)]
final class SendDraft extends GmailActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'draft_id' => '',
      'message_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['draft_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Draft ID'),
      '#description' => $this->t('Gmail draft ID to send. You can get this from the Create Draft action token or from a List Drafts action. Supports token replacement.'),
      '#default_value' => $this->configuration['draft_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['message_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sent Message Token Name'),
      '#description' => $this->t('Optional token name to store the sent message information. Accessible as [token_name:id], [token_name:thread_id], [token_name:label_ids], etc.'),
      '#default_value' => $this->configuration['message_token_name'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $draft_id = $this->tokenService->replacePlain($this->configuration['draft_id']);
    $message_token_name = $this->configuration['message_token_name'];

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('send draft');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($draft_id)) {
      $this->logger->error('Gmail send draft action: No draft ID provided.');
      return;
    }

    // Send the draft
    $sent_message = $this->gmailService->sendDraft($auth_type, $client_id, $draft_id);

    if (!$sent_message) {
      $this->logger->error('Gmail send draft action: Failed to send draft @draft_id using @auth_type.', [
        '@draft_id' => $draft_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    $this->logger->info('Gmail send draft action: Successfully sent draft @draft_id as message @message_id.', [
      '@draft_id' => $draft_id,
      '@message_id' => $sent_message['id'] ?? 'unknown',
    ]);

    // Store sent message in token if requested
    if (!empty($message_token_name)) {
      $this->tokenService->addTokenData($message_token_name, $sent_message);
    }
  }

}
