<?php

declare(strict_types=1);

namespace Drupal\eca_gmail\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Send emails via Gmail.
 */
#[Action(
  id: 'eca_gmail_send_email',
  label: new TranslatableMarkup('Gmail: Send Email'),
  category: new TranslatableMarkup('Gmail'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Send emails with HTML/plain text content and attachments via Gmail API with support for multiple recipients and custom headers.'),
  version_introduced: '1.0.0',
)]
final class SendEmail extends GmailActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'to' => '',
      'cc' => '',
      'bcc' => '',
      'subject' => '',
      'body_html' => '',
      'body_plain' => '',
      'reply_to' => '',
      'thread_id' => '',
      'message_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['to'] = [
      '#type' => 'textfield',
      '#title' => $this->t('To'),
      '#description' => $this->t('Recipient email addresses (comma-separated). Supports token replacement.'),
      '#default_value' => $this->configuration['to'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['cc'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CC'),
      '#description' => $this->t('CC email addresses (comma-separated). Supports token replacement.'),
      '#default_value' => $this->configuration['cc'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['bcc'] = [
      '#type' => 'textfield',
      '#title' => $this->t('BCC'),
      '#description' => $this->t('BCC email addresses (comma-separated). Supports token replacement.'),
      '#default_value' => $this->configuration['bcc'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#description' => $this->t('Email subject line. Supports token replacement.'),
      '#default_value' => $this->configuration['subject'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['body_html'] = [
      '#type' => 'textarea',
      '#title' => $this->t('HTML Body'),
      '#description' => $this->t('HTML email content. If both HTML and plain text are provided, a multipart email will be sent. Supports token replacement.'),
      '#default_value' => $this->configuration['body_html'],
      '#rows' => 8,
      '#eca_token_replacement' => TRUE,
    ];

    $form['body_plain'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Plain Text Body'),
      '#description' => $this->t('Plain text email content. Used as fallback for HTML emails or as the main content for plain text emails. Supports token replacement.'),
      '#default_value' => $this->configuration['body_plain'],
      '#rows' => 8,
      '#eca_token_replacement' => TRUE,
    ];

    $form['reply_to'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Reply-To'),
      '#description' => $this->t('Optional reply-to email address. Supports token replacement.'),
      '#default_value' => $this->configuration['reply_to'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['thread_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Thread ID'),
      '#description' => $this->t('Optional Gmail thread ID to reply to an existing conversation. Supports token replacement.'),
      '#default_value' => $this->configuration['thread_id'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['message_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Message Token Name'),
      '#description' => $this->t('Optional token name to store sent message information. Accessible as [token_name:message_id], [token_name:thread_id], etc.'),
      '#default_value' => $this->configuration['message_token_name'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $to = $this->tokenService->replacePlain($this->configuration['to']);
    $cc = $this->tokenService->replacePlain($this->configuration['cc']);
    $bcc = $this->tokenService->replacePlain($this->configuration['bcc']);
    $subject = $this->tokenService->replacePlain($this->configuration['subject']);
    $body_html = $this->tokenService->replace($this->configuration['body_html']);
    $body_plain = $this->tokenService->replace($this->configuration['body_plain']);
    $reply_to = $this->tokenService->replacePlain($this->configuration['reply_to']);
    $thread_id = $this->tokenService->replacePlain($this->configuration['thread_id'], [], ['clear' => TRUE]);
    $message_token_name = $this->configuration['message_token_name'];

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('send email');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($to) || empty($subject)) {
      $this->logger->error('Gmail send email action: Missing required fields (to, subject).');
      return;
    }

    // Validate email addresses
    $to_addresses = $this->parseEmailAddresses($to);
    foreach ($to_addresses as $email) {
      if (!$this->validateEmailAddress($email)) {
        $this->logger->error('Gmail send email action: Invalid email address in To field: @email', [
          '@email' => $email,
        ]);
        return;
      }
    }

    if (!empty($cc)) {
      $cc_addresses = $this->parseEmailAddresses($cc);
      foreach ($cc_addresses as $email) {
        if (!$this->validateEmailAddress($email)) {
          $this->logger->error('Gmail send email action: Invalid email address in CC field: @email', [
            '@email' => $email,
          ]);
          return;
        }
      }
    }

    if (!empty($bcc)) {
      $bcc_addresses = $this->parseEmailAddresses($bcc);
      foreach ($bcc_addresses as $email) {
        if (!$this->validateEmailAddress($email)) {
          $this->logger->error('Gmail send email action: Invalid email address in BCC field: @email', [
            '@email' => $email,
          ]);
          return;
        }
      }
    }

    if (empty($body_html) && empty($body_plain)) {
      $this->logger->error('Gmail send email action: Either HTML body or plain text body must be provided.');
      return;
    }


    // Prepare message parameters
    $message_params = [
      'to' => $to,
      'subject' => $subject,
    ];

    if (!empty($cc)) {
      $message_params['cc'] = $cc;
    }

    if (!empty($bcc)) {
      $message_params['bcc'] = $bcc;
    }

    if (!empty($body_html)) {
      $message_params['body_html'] = $body_html;
    }

    if (!empty($body_plain)) {
      $message_params['body_plain'] = $body_plain;
    }

    if (!empty($reply_to)) {
      $message_params['reply_to'] = $reply_to;
    }

    if (!empty($thread_id)) {
      $message_params['thread_id'] = $thread_id;
    }

    // Send the message
    $message_data = $this->gmailService->sendMessage($auth_type, $client_id, $message_params);

    if (!$message_data) {
      $this->logger->error('Gmail send email action: Failed to send message using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store message data in token if requested
    if (!empty($message_token_name)) {
      $this->tokenService->addTokenData($message_token_name, $message_data);
    }
  }

}
