<?php

declare(strict_types=1);

namespace Drupal\eca_gmail\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Set Gmail vacation auto-reply settings.
 */
#[Action(
  id: 'eca_gmail_set_auto_reply',
  label: new TranslatableMarkup('Gmail: Set Auto-Reply'),
  category: new TranslatableMarkup('Gmail'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Configure vacation/out-of-office responses with scheduled activation, contact filtering, and response tracking.'),
  version_introduced: '1.0.0',
)]
final class SetAutoReply extends GmailActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'enable_auto_reply' => '1',
      'response_subject' => '',
      'response_body_html' => '',
      'response_body_plain_text' => '',
      'start_time' => '',
      'end_time' => '',
      'restrict_to_contacts' => '0',
      'restrict_to_domain' => '0',
      'auto_reply_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['enable_auto_reply'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Enable Auto-Reply'),
      '#description' => $this->t('Set to "1", "true", or "yes" to enable auto-reply. Set to "0", "false", or "no" to disable. Supports token replacement.'),
      '#default_value' => $this->configuration['enable_auto_reply'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['response_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Response Subject'),
      '#description' => $this->t('Subject line for auto-reply messages. Supports token replacement.'),
      '#default_value' => $this->configuration['response_subject'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['response_body_html'] = [
      '#type' => 'textarea',
      '#title' => $this->t('HTML Response Body'),
      '#description' => $this->t('HTML content for auto-reply messages. If both HTML and plain text are provided, a multipart response will be sent. Supports token replacement.'),
      '#default_value' => $this->configuration['response_body_html'],
      '#rows' => 6,
      '#eca_token_replacement' => TRUE,
    ];

    $form['response_body_plain_text'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Plain Text Response Body'),
      '#description' => $this->t('Plain text content for auto-reply messages. Used as fallback for HTML responses or as the main content. Supports token replacement.'),
      '#default_value' => $this->configuration['response_body_plain_text'],
      '#rows' => 6,
      '#eca_token_replacement' => TRUE,
    ];

    $form['start_time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Start Time'),
      '#description' => $this->t('Optional start time for auto-reply. For date range fields use [entity:field_date_range:start_date], for Smart Date fields use [entity:field_smart_date:value], for regular datetime fields use [entity:field_date:date]. Also supports string formats ("2024-12-25 10:00", "December 25, 2024 10:00 AM"). Leave empty for immediate activation.'),
      '#default_value' => $this->configuration['start_time'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['end_time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('End Time'),
      '#description' => $this->t('Optional end time for auto-reply. For date range fields use [entity:field_date_range:end_date], for Smart Date fields use [entity:field_smart_date:end_value], for regular datetime fields use [entity:field_date:date]. Also supports string formats ("2024-12-31 18:00", "December 31, 2024 6:00 PM"). Leave empty for no expiration.'),
      '#default_value' => $this->configuration['end_time'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['restrict_to_contacts'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Restrict to Contacts'),
      '#description' => $this->t('Set to "1", "true", or "yes" to only send auto-replies to people in your contacts. Supports token replacement.'),
      '#default_value' => $this->configuration['restrict_to_contacts'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['restrict_to_domain'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Restrict to Domain'),
      '#description' => $this->t('Set to "1", "true", or "yes" to only send auto-replies to people in your domain (G Workspace). Supports token replacement.'),
      '#default_value' => $this->configuration['restrict_to_domain'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['auto_reply_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Auto-Reply Token Name'),
      '#description' => $this->t('Optional token name to store auto-reply settings. Accessible as [token_name:enabled], [token_name:start_time], [token_name:end_time], etc.'),
      '#default_value' => $this->configuration['auto_reply_token_name'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $enable_auto_reply = $this->tokenService->replacePlain($this->configuration['enable_auto_reply']);
    $response_subject = $this->tokenService->replacePlain($this->configuration['response_subject']);
    $response_body_html = $this->tokenService->replace($this->configuration['response_body_html']);
    $response_body_plain_text = $this->tokenService->replace($this->configuration['response_body_plain_text']);
    $start_time = $this->tokenService->replacePlain($this->configuration['start_time'], [], ['clear' => TRUE]);
    $end_time = $this->tokenService->replacePlain($this->configuration['end_time'], [], ['clear' => TRUE]);
    $restrict_to_contacts = $this->tokenService->replacePlain($this->configuration['restrict_to_contacts']);
    $restrict_to_domain = $this->tokenService->replacePlain($this->configuration['restrict_to_domain']);
    $auto_reply_token_name = $this->configuration['auto_reply_token_name'];

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('set auto-reply');
    if (!$auth_type || !$client_id) {
      return;
    }

    // Parse boolean values
    $enable_auto_reply_bool = !empty($enable_auto_reply);
    $restrict_to_contacts_bool = !empty($restrict_to_contacts);
    $restrict_to_domain_bool = !empty($restrict_to_domain);

    // Prepare vacation parameters
    $vacation_params = [
      'enable_auto_reply' => $enable_auto_reply_bool,
    ];

    if (!empty($response_subject)) {
      $vacation_params['response_subject'] = $response_subject;
    }

    if (!empty($response_body_html)) {
      $vacation_params['response_body_html'] = $response_body_html;
    }

    if (!empty($response_body_plain_text)) {
      $vacation_params['response_body_plain_text'] = $response_body_plain_text;
    }

    // Add start and end times (service will handle parsing and formatting)
    if (!empty($start_time)) {
      $vacation_params['start_time'] = $start_time;
    }

    if (!empty($end_time)) {
      $vacation_params['end_time'] = $end_time;
    }

    $vacation_params['restrict_to_contacts'] = $restrict_to_contacts_bool;
    $vacation_params['restrict_to_domain'] = $restrict_to_domain_bool;

    // Set auto-reply
    $auto_reply_data = $this->gmailService->setAutoReply($auth_type, $client_id, $vacation_params);

    if (!$auto_reply_data) {
      $this->logger->error('Gmail set auto-reply action: Failed to set auto-reply using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store auto-reply data in token if requested
    if (!empty($auto_reply_token_name)) {
      $this->tokenService->addTokenData($auto_reply_token_name, $auto_reply_data);
    }
  }

}
