<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\eca_google\DateTimeFormattingTrait;
use Drupal\eca_google\GoogleApiService;
use Google\Service\Calendar;
use Google\Service\Calendar\Event;
use Google\Service\Calendar\FreeBusyRequest;

/**
 * Service for Google Calendar API operations.
 */
class GoogleCalendarService {

  use StringTranslationTrait;
  use DateTimeFormattingTrait;

  public const string CALENDAR_PRIMARY = 'primary';

  public const string VISIBILITY_DEFAULT = 'default';
  public const string VISIBILITY_PUBLIC = 'public';
  public const string VISIBILITY_PRIVATE = 'private';
  public const string VISIBILITY_CONFIDENTIAL = 'confidential';

  public const string STATUS_CONFIRMED = 'confirmed';
  public const string STATUS_TENTATIVE = 'tentative';
  public const string STATUS_CANCELLED = 'cancelled';

  public const string ORDER_BY_START_TIME = 'startTime';
  public const string ORDER_BY_UPDATED = 'updated';

  public const string ACCESS_ROLE_FREE_BUSY_READER = 'freeBusyReader';
  public const string ACCESS_ROLE_READER = 'reader';
  public const string ACCESS_ROLE_WRITER = 'writer';
  public const string ACCESS_ROLE_OWNER = 'owner';

  public const string ACL_SCOPE_TYPE_DEFAULT = 'default';
  public const string ACL_SCOPE_TYPE_USER = 'user';
  public const string ACL_SCOPE_TYPE_GROUP = 'group';
  public const string ACL_SCOPE_TYPE_DOMAIN = 'domain';

  /**
   * The Google API service.
   */
  protected GoogleApiService $googleApiService;

  /**
   * The logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * @param GoogleApiService $google_api_service
   * @param LoggerChannelFactoryInterface $logger_factory
   */
  public function __construct(
    GoogleApiService $google_api_service,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->googleApiService = $google_api_service;
    $this->logger = $logger_factory->get('eca_google_calendar');
  }

  /**
   * Gets a configured Google Calendar service instance.
   *
   * @param string $auth_type
   *   The authentication type ('api_client' or 'service_account').
   * @param string $client_id
   *   The Google API client entity ID.
   *
   * @return \Google\Service\Calendar|null
   *   The Google Calendar service instance or NULL on failure.
   */
  public function getCalendarService(string $auth_type, string $client_id): ?Calendar {
    $service = $this->googleApiService->getService('calendar', $auth_type, $client_id);
    return $service instanceof Calendar ? $service : NULL;
  }

  /**
   * Validates Google Calendar API access.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   *
   * @return bool
   *   TRUE if Calendar API is accessible, FALSE otherwise.
   */
  public function validateCalendarAccess(string $auth_type, string $client_id): bool {
    return $this->googleApiService->validateApiAccess('calendar', $auth_type, $client_id);
  }

  /**
   * Gets visibility options for events.
   *
   * @return array
   *   Array of visibility options.
   */
  public function getVisibilityOptions(): array {
    return [
      self::VISIBILITY_DEFAULT => $this->t('Default - Use calendar default'),
      self::VISIBILITY_PUBLIC => $this->t('Public - Event details visible to all'),
      self::VISIBILITY_PRIVATE => $this->t('Private - Event details visible to attendees'),
      self::VISIBILITY_CONFIDENTIAL => $this->t('Confidential - Only free/busy information visible'),
    ];
  }

  /**
   * Gets status options for events.
   *
   * @return array
   *   Array of status options.
   */
  public function getEventStatusOptions(): array {
    return [
      self::STATUS_CONFIRMED => $this->t('Confirmed'),
      self::STATUS_TENTATIVE => $this->t('Tentative'),
      self::STATUS_CANCELLED => $this->t('Cancelled'),
    ];
  }

  /**
   * Gets minimum access role options for calendar filtering.
   *
   * @return array
   *   Array of access role options.
   */
  public function getAccessRoleOptions(): array {
    return [
      self::ACCESS_ROLE_FREE_BUSY_READER => $this->t('Free/Busy Reader - Can view free/busy information'),
      self::ACCESS_ROLE_READER => $this->t('Reader - Can view event details'),
      self::ACCESS_ROLE_WRITER => $this->t('Writer - Can create and edit events'),
      self::ACCESS_ROLE_OWNER => $this->t('Owner - Full calendar control'),
    ];
  }

  /**
   * Gets ACL scope type options for calendar access management.
   *
   * @return array
   *   Array of ACL scope type options.
   */
  public function getAclScopeTypeOptions(): array {
    return [
      self::ACL_SCOPE_TYPE_DEFAULT => $this->t('Default - Public access'),
      self::ACL_SCOPE_TYPE_USER => $this->t('User - Single user access'),
      self::ACL_SCOPE_TYPE_GROUP => $this->t('Group - Group access'),
      self::ACL_SCOPE_TYPE_DOMAIN => $this->t('Domain - Domain-wide access'),
    ];
  }

  /**
   * Creates a new calendar event.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param array $config
   *   Event configuration options.
   *
   * @return array|null
   *   Event details or NULL on failure.
   */
  public function createEvent(string $auth_type, string $client_id, array $config = []): ?array {
    $service = $this->getCalendarService($auth_type, $client_id);

    if (!$service) {
      return NULL;
    }

    try {
      $event = new Event();

      // Basic event properties - assume validation done by action
      $event->setSummary($config['summary']);

      if (!empty($config['description'])) {
        $event->setDescription($config['description']);
      }

      if (!empty($config['location'])) {
        $event->setLocation($config['location']);
      }

      // Configure start and end times
      $this->configureEventDateTime($event, $config);

      // Configure attendees
      if (!empty($config['attendees'])) {
        $this->configureEventAttendees($event, $config['attendees']);
      }

      // Visibility and status - only set if not default
      if (!empty($config['visibility']) && $config['visibility'] !== self::VISIBILITY_DEFAULT) {
        $event->setVisibility($config['visibility']);
      }

      if (!empty($config['status'])) {
        $event->setStatus($config['status']);
      }


      // Configure reminders using utility method
      if (!empty($config['reminders'])) {
        $this->configureEventReminders($event, $config['reminders']);
      }

      // Configure video conference based on user preferences
      if (!empty($config['use_meet'])) {
        $this->configureGoogleMeet($event);
      } elseif (!empty($config['conference_url'])) {
        $this->configureExternalConference($event, $config['conference_url']);
      }

      $calendar_id = $config['calendar_id'] ?? self::CALENDAR_PRIMARY;

      /** @var Event $result */
      $params = [];
      // Add conferenceDataVersion when conference data is present
      if ($event->getConferenceData()) {
        $params['conferenceDataVersion'] = 1;
      }
      $result = $service->events->insert($calendar_id, $event, $params);

      return $this->formatEventResult($result, TRUE);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create Google Calendar event: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Updates an existing calendar event.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param string $event_id
   *   The event ID to update.
   * @param array $config
   *   Event configuration options.
   *
   * @return array|null
   *   Updated event details or NULL on failure.
   */
  public function updateEvent(string $auth_type, string $client_id, string $event_id, array $config = []): ?array {
    $service = $this->getCalendarService($auth_type, $client_id);

    if (!$service) {
      return NULL;
    }

    try {
      $calendar_id = $config['calendar_id'] ?? self::CALENDAR_PRIMARY;

      // First get the existing event
      $event = $service->events->get($calendar_id, $event_id);

      // Update properties if provided
      if (isset($config['summary'])) {
        $event->setSummary($config['summary']);
      }

      if (isset($config['description'])) {
        $event->setDescription($config['description']);
      }

      if (isset($config['location'])) {
        $event->setLocation($config['location']);
      }

      // Update start/end times
      $this->configureEventDateTimeUpdate($event, $config, 'start');
      $this->configureEventDateTimeUpdate($event, $config, 'end');

      // Update attendees
      if (isset($config['attendees'])) {
        $this->configureEventAttendees($event, $config['attendees']);
      }

      if (isset($config['visibility'])) {
        $event->setVisibility($config['visibility']);
      }

      if (isset($config['status'])) {
        $event->setStatus($config['status']);
      }

      // Update reminders using utility method
      if (isset($config['reminders'])) {
        $this->configureEventReminders($event, $config['reminders']);
      }

      // Configure video conference based on user preferences
      if (!empty($config['use_meet'])) {
        // Only create Google Meet if event doesn't already have one
        $existing_conference = $event->getConferenceData();
        $has_google_meet = FALSE;

        // Check if it's specifically a Google Meet by looking for meet.google.com URLs
        if ($existing_conference && $existing_conference->getEntryPoints()) {
          foreach ($existing_conference->getEntryPoints() as $entry_point) {
            if ($entry_point->getUri() && strpos($entry_point->getUri(), 'meet.google.com') !== FALSE) {
              $has_google_meet = TRUE;
              break;
            }
          }
        }

        if (!$has_google_meet) {
          $this->configureGoogleMeet($event);
        }
      } elseif (isset($config['conference_url'])) {
        $this->configureExternalConference($event, $config['conference_url']);
      } else {
        // Handle conference removal when use_meet is false and no external URL
        if (isset($config['use_meet']) && !$config['use_meet']) {
          // Create empty conference data to remove existing conference
          $event->setConferenceData(new Calendar\ConferenceData());
        }
      }

      /** @var Event $result */
      $params = [];
      // Add conferenceDataVersion when conference data is present
      if ($event->getConferenceData()) {
        $params['conferenceDataVersion'] = 1;
      }
      $result = $service->events->update($calendar_id, $event_id, $event, $params);

      return $this->formatEventResult($result, FALSE);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to update Google Calendar event: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Deletes a calendar event.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param string $event_id
   *   The event ID to delete.
   * @param array $config
   *   Configuration options.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function deleteEvent(string $auth_type, string $client_id, string $event_id, array $config = []): bool {
    $service = $this->getCalendarService($auth_type, $client_id);

    if (!$service) {
      return FALSE;
    }

    try {
      $calendar_id = $config['calendar_id'] ?? self::CALENDAR_PRIMARY;
      $service->events->delete($calendar_id, $event_id);
      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to delete Google Calendar event: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Lists calendar events with filtering options.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param array $config
   *   Filtering and query options.
   *
   * @return array|null
   *   Array of events or NULL on failure.
   */
  public function listEvents(string $auth_type, string $client_id, array $config = []): ?array {
    $service = $this->getCalendarService($auth_type, $client_id);

    if (!$service) {
      return NULL;
    }

    try {
      $calendar_id = $config['calendar_id'] ?? self::CALENDAR_PRIMARY;
      $params = [];

      if (!empty($config['time_min'])) {
        $params['timeMin'] = $config['time_min'];
      }

      if (!empty($config['time_max'])) {
        $params['timeMax'] = $config['time_max'];
      }

      if (!empty($config['q'])) {
        $params['q'] = $config['q'];
      }

      if (!empty($config['max_results'])) {
        $params['maxResults'] = (int) $config['max_results'];
      }

      if (!empty($config['order_by'])) {
        $params['orderBy'] = $config['order_by'];
      }

      if (!empty($config['single_events'])) {
        $params['singleEvents'] = TRUE;
      }

      $events = $service->events->listEvents($calendar_id, $params);
      $result_events = [];

      foreach ($events->getItems() as $event) {
        $result_events[] = [
          'event_id' => $event->getId(),
          'summary' => $event->getSummary(),
          'description' => $event->getDescription(),
          'location' => $event->getLocation(),
          'start' => $event->getStart() ? $event->getStart()->getDateTime() : NULL,
          'end' => $event->getEnd() ? $event->getEnd()->getDateTime() : NULL,
          'status' => $event->getStatus(),
          'event_url' => $event->getHtmlLink(),
          'attendees_count' => $event->getAttendees() ? count($event->getAttendees()) : 0,
        ];
      }

      return [
        'events' => $result_events,
        'count' => count($result_events),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to list Google Calendar events: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets a specific calendar event.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param string $event_id
   *   The event ID to retrieve.
   * @param array $config
   *   Configuration options.
   *
   * @return array|null
   *   Event details or NULL on failure.
   */
  public function getEvent(string $auth_type, string $client_id, string $event_id, array $config = []): ?array {
    $service = $this->getCalendarService($auth_type, $client_id);

    if (!$service) {
      return NULL;
    }

    try {
      $calendar_id = $config['calendar_id'] ?? self::CALENDAR_PRIMARY;
      $event = $service->events->get($calendar_id, $event_id);

      $attendees = [];
      if ($event->getAttendees()) {
        foreach ($event->getAttendees() as $attendee) {
          $attendees[] = [
            'email' => $attendee->getEmail(),
            'display_name' => $attendee->getDisplayName(),
            'response_status' => $attendee->getResponseStatus(),
            'organizer' => $attendee->getOrganizer(),
          ];
        }
      }

      $result = [
        'event_id' => $event->getId(),
        'summary' => $event->getSummary(),
        'description' => $event->getDescription(),
        'location' => $event->getLocation(),
        'start' => $event->getStart() ? $event->getStart()->getDateTime() : NULL,
        'end' => $event->getEnd() ? $event->getEnd()->getDateTime() : NULL,
        'status' => $event->getStatus(),
        'visibility' => $event->getVisibility(),
        'event_url' => $event->getHtmlLink(),
        'attendees' => $attendees,
        'organizer' => $event->getOrganizer() ? [
          'email' => $event->getOrganizer()->getEmail(),
          'display_name' => $event->getOrganizer()->getDisplayName(),
        ] : NULL,
        'created' => $event->getCreated(),
        'updated' => $event->getUpdated(),
      ];

      // Include Google Meet conference data if available
      $conference_data = $event->getConferenceData();
      if ($conference_data) {
        $entry_points = $conference_data->getEntryPoints();
        if ($entry_points) {
          foreach ($entry_points as $entry_point) {
            if ($entry_point->getEntryPointType() === 'video') {
              $result['meet_url'] = $entry_point->getUri();
              break;
            }
          }
        }
        if ($conference_data->getConferenceId()) {
          $result['meet_id'] = $conference_data->getConferenceId();
        }
      }

      return $result;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get Google Calendar event: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Checks free/busy information for calendars.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param array $config
   *   Free/busy query configuration.
   *
   * @return array|null
   *   Free/busy data or NULL on failure.
   */
  public function checkFreeBusy(string $auth_type, string $client_id, array $config = []): ?array {
    $service = $this->getCalendarService($auth_type, $client_id);

    if (!$service) {
      return NULL;
    }

    try {
      $request = new FreeBusyRequest();

      if (!empty($config['time_min'])) {
        $request->setTimeMin($config['time_min']);
      }

      if (!empty($config['time_max'])) {
        $request->setTimeMax($config['time_max']);
      }

      if (!empty($config['items'])) {
        $items = [];
        foreach ($config['items'] as $calendar_id) {
          $item = new Calendar\FreeBusyRequestItem();
          $item->setId($calendar_id);
          $items[] = $item;
        }
        $request->setItems($items);
      }

      $response = $service->freebusy->query($request);

      $calendars = [];
      foreach ($response->getCalendars() as $calendar_id => $calendar_data) {
        $busy_times = [];
        foreach ($calendar_data->getBusy() as $busy_period) {
          $busy_times[] = [
            'start' => $busy_period->getStart(),
            'end' => $busy_period->getEnd(),
          ];
        }

        $calendars[$calendar_id] = [
          'busy' => $busy_times,
          'errors' => $calendar_data->getErrors(),
        ];
      }

      return [
        'calendars' => $calendars,
        'time_min' => $response->getTimeMin(),
        'time_max' => $response->getTimeMax(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to check Google Calendar free/busy: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Lists calendars accessible to the authenticated user.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param array $config
   *   Configuration options including filtering and pagination.
   *
   * @return array|null
   *   Array of calendar data or NULL on failure.
   */
  public function listCalendars(string $auth_type, string $client_id, array $config = []): ?array {
    $service = $this->getCalendarService($auth_type, $client_id);

    if (!$service) {
      return NULL;
    }

    try {
      $params = [];

      if (!empty($config['max_results'])) {
        $params['maxResults'] = (int) $config['max_results'];
      }

      if (!empty($config['min_access_role'])) {
        $params['minAccessRole'] = $config['min_access_role'];
      }

      if (!empty($config['show_deleted'])) {
        $params['showDeleted'] = (bool) $config['show_deleted'];
      }

      if (!empty($config['show_hidden'])) {
        $params['showHidden'] = (bool) $config['show_hidden'];
      }

      if (!empty($config['page_token'])) {
        $params['pageToken'] = $config['page_token'];
      }

      if (!empty($config['sync_token'])) {
        $params['syncToken'] = $config['sync_token'];
      }

      $calendar_list = $service->calendarList->listCalendarList($params);
      $result_calendars = [];

      foreach ($calendar_list->getItems() as $calendar) {
        $result_calendars[] = [
          'calendar_id' => $calendar->getId(),
          'summary' => $calendar->getSummary(),
          'description' => $calendar->getDescription(),
          'location' => $calendar->getLocation(),
          'time_zone' => $calendar->getTimeZone(),
          'summary_override' => $calendar->getSummaryOverride(),
          'color_id' => $calendar->getColorId(),
          'background_color' => $calendar->getBackgroundColor(),
          'foreground_color' => $calendar->getForegroundColor(),
          'access_role' => $calendar->getAccessRole(),
          'primary' => $calendar->getPrimary() ? TRUE : FALSE,
          'selected' => $calendar->getSelected() !== FALSE,
          'hidden' => $calendar->getHidden() ? TRUE : FALSE,
          'deleted' => $calendar->getDeleted() ? TRUE : FALSE,
          'default_reminders' => $calendar->getDefaultReminders() ? array_map(function($reminder) {
            return [
              'method' => $reminder->getMethod(),
              'minutes' => $reminder->getMinutes(),
            ];
          }, $calendar->getDefaultReminders()) : [],
        ];
      }

      return [
        'calendars' => $result_calendars,
        'count' => count($result_calendars),
        'next_page_token' => $calendar_list->getNextPageToken(),
        'next_sync_token' => $calendar_list->getNextSyncToken(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to list Google Calendars: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Creates a new calendar.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param array $config
   *   Calendar creation configuration.
   *
   * @return array|null
   *   Calendar data or NULL on failure.
   */
  public function createCalendar(string $auth_type, string $client_id, array $config = []): ?array {
    $service = $this->getCalendarService($auth_type, $client_id);

    if (!$service) {
      return NULL;
    }

    try {
      $calendar = new Calendar\Calendar();

      // Required field
      if (!empty($config['summary'])) {
        $calendar->setSummary($config['summary']);
      }

      // Optional fields
      if (!empty($config['description'])) {
        $calendar->setDescription($config['description']);
      }

      if (!empty($config['location'])) {
        $calendar->setLocation($config['location']);
      }

      if (!empty($config['time_zone'])) {
        $calendar->setTimeZone($config['time_zone']);
      }

      $created_calendar = $service->calendars->insert($calendar);

      return [
        'calendar_id' => $created_calendar->getId(),
        'summary' => $created_calendar->getSummary(),
        'description' => $created_calendar->getDescription(),
        'location' => $created_calendar->getLocation(),
        'time_zone' => $created_calendar->getTimeZone(),
        'etag' => $created_calendar->getEtag(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create Google Calendar: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Updates an existing calendar.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param string $calendar_id
   *   The calendar ID to update.
   * @param array $config
   *   Calendar update configuration.
   *
   * @return array|null
   *   Updated calendar data or NULL on failure.
   */
  public function updateCalendar(string $auth_type, string $client_id, string $calendar_id, array $config = []): ?array {
    $service = $this->getCalendarService($auth_type, $client_id);

    if (!$service) {
      return NULL;
    }

    try {
      // First get the current calendar to preserve existing data
      $existing_calendar = $service->calendars->get($calendar_id);

      // Update only the provided fields
      if (!empty($config['summary'])) {
        $existing_calendar->setSummary($config['summary']);
      }

      if (!empty($config['description'])) {
        $existing_calendar->setDescription($config['description']);
      }

      if (!empty($config['location'])) {
        $existing_calendar->setLocation($config['location']);
      }

      if (!empty($config['time_zone'])) {
        $existing_calendar->setTimeZone($config['time_zone']);
      }

      $updated_calendar = $service->calendars->update($calendar_id, $existing_calendar);

      return [
        'calendar_id' => $updated_calendar->getId(),
        'summary' => $updated_calendar->getSummary(),
        'description' => $updated_calendar->getDescription(),
        'location' => $updated_calendar->getLocation(),
        'time_zone' => $updated_calendar->getTimeZone(),
        'etag' => $updated_calendar->getEtag(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to update Google Calendar: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Deletes a secondary calendar.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param string $calendar_id
   *   The calendar ID to delete.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function deleteCalendar(string $auth_type, string $client_id, string $calendar_id): bool {
    $service = $this->getCalendarService($auth_type, $client_id);

    if (!$service) {
      return FALSE;
    }

    try {
      // Prevent deletion of primary calendar
      if ($calendar_id === self::CALENDAR_PRIMARY || $calendar_id === 'primary') {
        $this->logger->error('Cannot delete primary calendar.');
        return FALSE;
      }

      $service->calendars->delete($calendar_id);
      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to delete Google Calendar: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Manages calendar access control (ACL) rules.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param string $calendar_id
   *   The calendar ID to manage access for.
   * @param array $config
   *   Access management configuration.
   *
   * @return array|null
   *   Access rule data or NULL on failure.
   */
  public function manageCalendarAccess(string $auth_type, string $client_id, string $calendar_id, array $config = []): ?array {
    $service = $this->getCalendarService($auth_type, $client_id);

    if (!$service) {
      return NULL;
    }

    try {
      $operation = $config['operation'] ?? 'add';
      $role = $config['role'] ?? self::ACCESS_ROLE_READER;
      $scope_type = $config['scope_type'] ?? self::ACL_SCOPE_TYPE_USER;
      $scope_value = $config['scope_value'] ?? '';
      $send_notifications = $config['send_notifications'] ?? FALSE;

      switch ($operation) {
        case 'add':
          $acl = new Calendar\AclRule();
          $acl->setRole($role);

          $scope = new Calendar\AclRuleScope();
          $scope->setType($scope_type);
          if (!empty($scope_value)) {
            $scope->setValue($scope_value);
          }
          $acl->setScope($scope);

          $params = [];
          $params['sendNotifications'] = (bool) $send_notifications;
          $created_acl = $service->acl->insert($calendar_id, $acl, $params);

          return [
            'operation' => 'add',
            'acl_id' => $created_acl->getId(),
            'role' => $created_acl->getRole(),
            'scope_type' => $created_acl->getScope()->getType(),
            'scope_value' => $created_acl->getScope()->getValue(),
            'etag' => $created_acl->getEtag(),
          ];

        case 'list':
          $acl_list = $service->acl->listAcl($calendar_id);
          $acl_rules = [];

          foreach ($acl_list->getItems() as $acl_rule) {
            $acl_rules[] = [
              'acl_id' => $acl_rule->getId(),
              'role' => $acl_rule->getRole(),
              'scope_type' => $acl_rule->getScope()->getType(),
              'scope_value' => $acl_rule->getScope()->getValue(),
              'etag' => $acl_rule->getEtag(),
            ];
          }

          return [
            'operation' => 'list',
            'acl_rules' => $acl_rules,
            'count' => count($acl_rules),
          ];

        case 'update':
          $acl_id = $config['acl_id'] ?? '';
          if (empty($acl_id)) {
            $this->logger->error('ACL ID required for update operation.');
            return NULL;
          }

          // Get the existing ACL rule
          $existing_acl = $service->acl->get($calendar_id, $acl_id);

          // Update the role if provided
          if (!empty($role)) {
            $existing_acl->setRole($role);
          }

          // Update the scope if provided
          if (!empty($scope_type) || !empty($scope_value)) {
            $scope = $existing_acl->getScope();
            if (!empty($scope_type)) {
              $scope->setType($scope_type);
            }
            if (!empty($scope_value)) {
              $scope->setValue($scope_value);
            }
            $existing_acl->setScope($scope);
          }

          $params = [];
          $params['sendNotifications'] = (bool) $send_notifications;

          $updated_acl = $service->acl->update($calendar_id, $acl_id, $existing_acl, $params);

          return [
            'operation' => 'update',
            'acl_id' => $updated_acl->getId(),
            'role' => $updated_acl->getRole(),
            'scope_type' => $updated_acl->getScope()->getType(),
            'scope_value' => $updated_acl->getScope()->getValue(),
            'etag' => $updated_acl->getEtag(),
          ];

        case 'remove':
          $acl_id = $config['acl_id'] ?? '';
          if (empty($acl_id)) {
            $this->logger->error('ACL ID required for remove operation.');
            return NULL;
          }

          $service->acl->delete($calendar_id, $acl_id);

          return [
            'operation' => 'remove',
            'acl_id' => $acl_id,
            'success' => TRUE,
          ];

        default:
          $this->logger->error('Invalid operation: @operation', ['@operation' => $operation]);
          return NULL;
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to manage Google Calendar access: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Configures start and end times for a Google Calendar event.
   *
   * @param Calendar\Event $event
   *   The event object to configure.
   * @param array $config
   *   Configuration array containing start_time, end_time, timezone, etc.
   *
   * @return void
   */
  protected function configureEventDateTime(Event $event, array $config): void {
    $start = new Calendar\EventDateTime();
    $end = new Calendar\EventDateTime();

    // Check if this is an all-day event using existing utility method
    $is_all_day = isset($config['start_time_original'], $config['end_time_original'])
      ? $this->isAllDayEventRange($config['start_time_original'], $config['end_time_original'], $config['timezone'] ?? NULL)
      : FALSE;

    if ($is_all_day) {
      // For all-day events, use date instead of dateTime
      $start->setDate($config['start_time']);

      // For end date, if it's the same day as start, Google Calendar expects next day
      $start_date = $config['start_time'];
      $end_date = $config['end_time'];

      // If end date is same as start date, add one day for Google Calendar
      if ($start_date === $end_date) {
        $end_dt = new \DateTime($end_date);
        $end_dt->modify('+1 day');
        $end_date = $end_dt->format('Y-m-d');
      }

      $end->setDate($end_date);
    } else {
      // For timed events, use dateTime
      $start->setDateTime($config['start_time']);
      $end->setDateTime($config['end_time']);
      if (!empty($config['timezone'])) {
        $start->setTimeZone($config['timezone']);
        $end->setTimeZone($config['timezone']);
      }
    }

    $event->setStart($start);
    $event->setEnd($end);
  }

  /**
   * Configures individual start or end time for event updates.
   *
   * @param Calendar\Event $event
   *   The event object to configure.
   * @param array $config
   *   Configuration array.
   * @param string $time_type
   *   Either 'start' or 'end'.
   *
   * @return void
   */
  protected function configureEventDateTimeUpdate(Event $event, array $config, string $time_type): void {
    $time_key = $time_type . '_time';
    $original_key = $time_type . '_time_original';

    if (empty($config[$time_key])) {
      return;
    }

    $datetime = new Calendar\EventDateTime();

    // Check if this is an all-day event
    $is_all_day = isset($config['start_time_original'], $config['end_time_original'])
      ? $this->isAllDayEventRange($config['start_time_original'], $config['end_time_original'], $config['timezone'] ?? NULL)
      : FALSE;

    if ($is_all_day && isset($config[$original_key])) {
      if ($time_type === 'end') {
        // All-day end event - handle same-day adjustment
        $start_date = $config['start_time'];
        $end_date = $config['end_time'];
        if ($start_date === $end_date) {
          $end_dt = new \DateTime($end_date);
          $end_dt->modify('+1 day');
          $end_date = $end_dt->format('Y-m-d');
        }
        $datetime->setDate($end_date);
      } else {
        // All-day start event - use date format
        $datetime->setDate($config[$time_key]);
      }
    } else {
      // Timed event - use datetime format
      $datetime->setDateTime($config[$time_key]);
      if (!empty($config['timezone'])) {
        $datetime->setTimeZone($config['timezone']);
      }
    }

    if ($time_type === 'start') {
      $event->setStart($datetime);
    } else {
      $event->setEnd($datetime);
    }
  }

  /**
   * Configures event attendees.
   *
   * @param Calendar\Event $event
   *   The event object to configure.
   * @param array $attendees_emails
   *   Array of email addresses.
   *
   * @return void
   */
  protected function configureEventAttendees(Event $event, array $attendees_emails): void {
    if (empty($attendees_emails)) {
      return;
    }

    $attendees = [];
    foreach ($attendees_emails as $email) {
      $attendee = new Calendar\EventAttendee();
      $attendee->setEmail($email);
      $attendees[] = $attendee;
    }
    $event->setAttendees($attendees);
  }

  /**
   * Configures event reminders.
   *
   * @param Calendar\Event $event
   *   The event object to configure.
   * @param array $reminders
   *   Array of reminder arrays with 'method' and 'minutes' keys.
   *
   * @return void
   */
  protected function configureEventReminders(Event $event, array $reminders): void {
    if (empty($reminders)) {
      return;
    }

    $reminders_config = new Calendar\EventReminders();
    $reminders_config->setUseDefault(FALSE);
    $reminder_overrides = [];
    foreach ($reminders as $reminder) {
      $override = new Calendar\EventReminder();
      $override->setMethod($reminder['method']);
      $override->setMinutes($reminder['minutes']);
      $reminder_overrides[] = $override;
    }
    $reminders_config->setOverrides($reminder_overrides);
    $event->setReminders($reminders_config);
  }

  /**
   * Configures Google Meet video conference for an event.
   *
   * @param \Google\Service\Calendar\Event $event
   *   The event object to configure.
   *
   * @return void
   */
  protected function configureGoogleMeet(Event $event): void {
    $conference_request = new Calendar\ConferenceData();
    $create_request = new Calendar\CreateConferenceRequest();

    // Generate unique request ID for conference creation
    $create_request->setRequestId(uniqid('meet_', TRUE));

    // Set conference solution to Google Meet
    $solution_key = new Calendar\ConferenceSolutionKey();
    $solution_key->setType('hangoutsMeet');
    $create_request->setConferenceSolutionKey($solution_key);

    $conference_request->setCreateRequest($create_request);
    $event->setConferenceData($conference_request);
  }

  /**
   * Configures external video conference for an event.
   *
   * @param \Google\Service\Calendar\Event $event
   *   The event object to configure.
   * @param string $conference_url
   *   The external conference URL (e.g., Zoom, Teams, etc.).
   *
   * @return void
   */
  protected function configureExternalConference(Event $event, string $conference_url): void {
    if (empty($conference_url)) {
      return;
    }

    $conference_data = new Calendar\ConferenceData();

    // Create conference solution for external/third-party providers
    $solution = new Calendar\ConferenceSolution();
    $solution_key = new Calendar\ConferenceSolutionKey();
    $solution_key->setType('addOn'); // Required for third-party conference providers
    $solution->setKey($solution_key);
    $solution->setName('Video Conference'); // Generic name for external conferences
    $conference_data->setConferenceSolution($solution);

    // Create entry point for external URL
    $entry_point = new Calendar\EntryPoint();
    $entry_point->setEntryPointType('video');
    $entry_point->setUri($conference_url);
    $entry_point->setLabel('Video Conference');

    $conference_data->setEntryPoints([$entry_point]);

    $event->setConferenceData($conference_data);
  }


  /**
   * Formats event result data into standardized array.
   *
   * @param Calendar\Event $result
   *   The event result from Google Calendar API.
   * @param bool $include_created
   *   Whether to include created timestamp (for create operations).
   *
   * @return array
   *   Standardized event data array.
   */
  protected function formatEventResult(Event $result, bool $include_created = FALSE): array {
    $formatted = [
      'event_id' => $result->getId(),
      'event_url' => $result->getHtmlLink(),
      'status' => $result->getStatus(),
      'updated' => $result->getUpdated(),
    ];

    if ($include_created) {
      $formatted['created'] = $result->getCreated();
    }

    // Include Google Meet conference data if available
    $conference_data = $result->getConferenceData();
    if ($conference_data) {
      $entry_points = $conference_data->getEntryPoints();
      if ($entry_points) {
        foreach ($entry_points as $entry_point) {
          if ($entry_point->getEntryPointType() === 'video') {
            $formatted['meet_url'] = $entry_point->getUri();
            break;
          }
        }
      }
      if ($conference_data->getConferenceId()) {
        $formatted['meet_id'] = $conference_data->getConferenceId();
      }
    }

    return $formatted;
  }

  /**
   * Parses attendees text into array of email addresses.
   *
   * @param string $attendees_text
   *   Text with email addresses, one per line.
   *
   * @return array
   *   Array of valid email addresses.
   */
  public function parseAttendees(string $attendees_text): array {
    $attendees = [];
    if (!empty($attendees_text)) {
      $attendee_lines = explode("\n", trim($attendees_text));
      foreach ($attendee_lines as $line) {
        $email = trim($line);
        if (!empty($email) && filter_var($email, FILTER_VALIDATE_EMAIL)) {
          $attendees[] = $email;
        }
      }
    }
    return $attendees;
  }

  /**
   * Checks if both start and end times represent an all-day event.
   *
   * @param mixed $start_time
   *   The start datetime value.
   * @param mixed $end_time
   *   The end datetime value.
   * @param string|null $timezone
   *   Optional timezone to check against.
   *
   * @return bool
   *   TRUE if both times qualify as all-day event times.
   */
  public function isAllDayEventRange(mixed $start_time, mixed $end_time, ?string $timezone = NULL): bool {
    $start_is_all_day = $this->isAllDayEvent($start_time, $timezone);
    $end_is_all_day = $this->isAllDayEvent($end_time, $timezone);
    return $start_is_all_day && $end_is_all_day;
  }

  /**
   * Parses reminder text into structured array.
   *
   * @param string $reminders_text
   *   Text with reminders, one per line in format "METHOD MINUTES".
   *
   * @return array
   *   Array of reminder arrays with 'method' and 'minutes' keys.
   */
  public function parseReminders(string $reminders_text): array {
    $reminders = [];
    if (!empty($reminders_text)) {
      $reminder_lines = explode("\n", trim($reminders_text));
      foreach ($reminder_lines as $line) {
        $line = trim($line);
        if (!empty($line)) {
          $parts = explode(' ', $line, 2);
          if (count($parts) === 2) {
            $method = trim($parts[0]);
            $minutes = trim($parts[1]);
            if (in_array($method, ['popup', 'email']) && is_numeric($minutes)) {
              $reminders[] = [
                'method' => $method,
                'minutes' => (int) $minutes,
              ];
            }
          }
        }
      }
    }
    return $reminders;
  }

  /**
   * Checks if a datetime represents an all-day event.
   *
   * @param mixed $datetime
   *   The datetime value to check.
   * @param string|null $timezone
   *   Optional timezone to check against.
   *
   * @return bool
   *   TRUE if this appears to be an all-day event.
   */
  public function isAllDayEvent(mixed $datetime, ?string $timezone = NULL): bool {
    if (empty($datetime)) {
      return FALSE;
    }

    // Check for Smart Date all-day format (time is 00:00:00)
    if (is_string($datetime)) {
      // Check if it's a timestamp
      if (is_numeric($datetime)) {
        return $this->isAllDayTimestamp((int) $datetime, $timezone);
      }

      // Check if time component is 00:00:00 or missing
      if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $datetime)) {
        return TRUE; // Date-only format
      }
      if (preg_match('/^\d{4}-\d{2}-\d{2}[\sT]00:00:00/', $datetime)) {
        return TRUE; // DateTime with 00:00:00 time
      }
    }

    // Check DateTime object
    if ($datetime instanceof \DateTimeInterface) {
      $time = $datetime->format('H:i:s');
      return $time === '00:00:00';
    }

    // Check Drupal field array
    if (is_array($datetime) && isset($datetime['date'])) {
      return $this->isAllDayEvent($datetime['date']);
    }

    return FALSE;
  }

  /**
   * Checks if a timestamp represents midnight in the site timezone.
   *
   * @param int $timestamp
   *   The timestamp to check.
   * @param string|null $timezone
   *   Optional timezone to use instead of site default.
   *
   * @return bool
   *   TRUE if this timestamp represents midnight in the specified timezone.
   */
  protected function isAllDayTimestamp(int $timestamp, ?string $timezone = NULL): bool {
    // Use provided timezone or fall back to site's default timezone
    $check_timezone = $timezone ?: (\Drupal::config('system.date')->get('timezone.default') ?: date_default_timezone_get());

    try {
      $dt = new \DateTime('@' . $timestamp);
      $dt->setTimezone(new \DateTimeZone($check_timezone));

      // Check if it's midnight (start of day) or 11:59:59 PM (end of day) in the site timezone
      $time = $dt->format('H:i:s');
      return $time === '00:00:00' || $time === '23:59:00' || $time === '23:59:59';
    }
    catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   * Formats datetime specifically for all-day events (date-only).
   *
   * @param mixed $datetime
   *   The datetime value to format.
   * @param string|null $timezone
   *   Optional timezone identifier.
   *
   * @return string|null
   *   Date-only formatted string (Y-m-d) or NULL on failure.
   */
  public function formatDateForAllDay(mixed $datetime, ?string $timezone = NULL): ?string {
    return $this->formatDateTimeForApi($datetime, $timezone, TRUE);
  }


}
