<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_google_calendar\GoogleCalendarService;

/**
 * Add access permissions to a Google Calendar.
 */
#[Action(
  id: 'eca_google_calendar_add_access',
  label: new TranslatableMarkup('Google Calendar: Add Access'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Grants calendar access permissions to users, groups, or domains with specified roles. Supports email notifications about the sharing change.'),
  version_introduced: '1.0.0',
)]
final class AddCalendarAccess extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'calendar_id' => '',
      'role' => GoogleCalendarService::ACCESS_ROLE_READER,
      'scope_type' => GoogleCalendarService::ACL_SCOPE_TYPE_USER,
      'scope_value' => '',
      'send_notifications' => FALSE,
      'access_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['calendar_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar ID'),
      '#description' => $this->t('The ID of the calendar to grant access to. Use "primary" for the user\'s primary calendar.'),
      '#default_value' => $this->configuration['calendar_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['role'] = [
      '#type' => 'select',
      '#title' => $this->t('Access Role'),
      '#description' => $this->t('The level of access to grant to the user/group/domain.'),
      '#options' => $this->googleCalendarService->getAccessRoleOptions(),
      '#default_value' => $this->configuration['role'],
      '#required' => TRUE,
    ];

    $form['scope_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Access Scope Type'),
      '#description' => $this->t('The type of entity to grant access to.'),
      '#options' => $this->googleCalendarService->getAclScopeTypeOptions(),
      '#default_value' => $this->configuration['scope_type'],
      '#required' => TRUE,
    ];

    $form['scope_value'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Email/Domain'),
      '#description' => $this->t('Email address for user/group access, domain name for domain access, or leave empty for default (public) access.'),
      '#default_value' => $this->configuration['scope_value'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['send_notifications'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send Notifications'),
      '#description' => $this->t('Whether to send email notifications about the sharing change.'),
      '#default_value' => $this->configuration['send_notifications'],
    ];

    $form['access_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Access Token Name'),
      '#description' => $this->t('Token name to store the access rule details. Accessible as [token_name:acl_id], [token_name:role], [token_name:scope_type], [token_name:scope_value].'),
      '#default_value' => $this->configuration['access_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $calendar_id = $this->tokenService->replacePlain($this->configuration['calendar_id']);
    $role = $this->configuration['role'];
    $scope_type = $this->configuration['scope_type'];
    $scope_value = $this->tokenService->replacePlain($this->configuration['scope_value'], [], ['clear' => TRUE]);
    $send_notifications = $this->configuration['send_notifications'];
    $access_token_name = $this->tokenService->replacePlain($this->configuration['access_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('add calendar access');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($calendar_id)) {
      $this->logger->error('Google Calendar add access action: Missing calendar_id.');
      return;
    }

    // Validate that scope value is provided for non-default scope types
    if ($scope_type !== 'default' && empty($scope_value)) {
      $this->logger->error('Google Calendar add access action: Email/domain value required for @scope_type access type.', [
        '@scope_type' => $scope_type,
      ]);
      return;
    }


    $config = [
      'operation' => 'add',
      'role' => $role,
      'scope_type' => $scope_type,
      'scope_value' => $scope_value,
      'send_notifications' => $send_notifications,
    ];

    $result = $this->googleCalendarService->manageCalendarAccess($auth_type, $client_id, $calendar_id, $config);

    if (!$result) {
      $this->logger->error('Google Calendar add access action: Failed to add access for calendar @calendar_id using @auth_type.', [
        '@calendar_id' => $calendar_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store access rule data
    if (!empty($access_token_name)) {
      $this->tokenService->addTokenData($access_token_name, $result);
    }

  }

}
