<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca_google\Plugin\Action\GoogleApiActionBase;
use Drupal\eca_google_calendar\GoogleCalendarService;
use Symfony\Component\DependencyInjection\ContainerInterface;

abstract class CalendarActionBase extends GoogleApiActionBase {

  /**
   * The Google Calendar service.
   */
  protected GoogleCalendarService $googleCalendarService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->googleCalendarService = $container->get('eca_google_calendar.google_calendar');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);
    $this->validateCalendarAccess($form, $form_state);
  }

  /**
   * Validates Calendar API access.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  protected function validateCalendarAccess(array $form, FormStateInterface $form_state): void {
    $auth_client_id = $form_state->getValue('auth_client_id');
    if (!empty($auth_client_id)) {
      [$auth_type, $client_id] = explode(':', $auth_client_id, 2);

      if (!in_array($auth_type, ['api_client', 'service_account'])) {
        $form_state->setErrorByName('auth_client_id', $this->t('Auth type must be "api_client" or "service_account".'));
        return;
      }

      if (!$this->googleCalendarService->validateCalendarAccess($auth_type, $client_id)) {
        $form_state->setErrorByName('auth_client_id', $this->t('The selected Google API client does not have access to Google Calendar API.'));
      }
    }
  }

  /**
   * Gets calendar options for selection.
   *
   * @return array
   *   Array of calendar options.
   */
  protected function getCalendarOptions(): array {
    return [
      GoogleCalendarService::CALENDAR_PRIMARY => $this->t('Primary Calendar'),
    ];
  }

  /**
   * Adds common calendar configuration form elements.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The modified form array.
   */
  protected function addCalendarConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['calendar_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar ID'),
      '#description' => $this->t('The calendar ID to use. Leave empty to use primary calendar.'),
      '#default_value' => $this->configuration['calendar_id'] ?? GoogleCalendarService::CALENDAR_PRIMARY,
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * Validates and parses authentication client ID.
   *
   * @param string $auth_client_id
   *   The auth client ID to validate and parse.
   * @param string $action_name
   *   The action name for error messages.
   *
   * @return array|null
   *   Array with 'auth_type' and 'client_id' keys, or NULL on failure.
   */

  /**
   * Validates formatted datetime values and logs errors.
   *
   * @param array $datetimes
   *   Array of datetime info ['name' => string, 'formatted' => string|null, 'original' => mixed].
   * @param string $action_name
   *   The action name for error messages.
   *
   * @return bool
   *   TRUE if all datetimes are valid, FALSE otherwise.
   */
  protected function validateFormattedDateTimes(array $datetimes, string $action_name): bool {
    foreach ($datetimes as $datetime_info) {
      if (!$datetime_info['formatted']) {
        $this->logger->error('Google Calendar @action action: Invalid @name format "@time".', [
          '@action' => $action_name,
          '@name' => $datetime_info['name'],
          '@time' => $datetime_info['original'],
        ]);
        return FALSE;
      }
    }
    return TRUE;
  }

}