<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Check free/busy status for Google Calendars.
 */
#[Action(
  id: 'eca_google_calendar_check_freebusy',
  label: new TranslatableMarkup('Google Calendar: Check Free/Busy'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Queries availability for scheduling coordination by checking free/busy status across multiple calendars within a time range.'),
  version_introduced: '1.0.0',
)]
final class CheckFreeBusy extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'time_min' => '',
      'time_max' => '',
      'calendar_ids' => '',
      'freebusy_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['time_min'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Start Time (From)'),
      '#description' => $this->t('Lower bound for free/busy query. For date range fields use [entity:field_date_range:start_date], for Smart Date fields use [entity:field_smart_date:value], for regular datetime fields use [entity:field_date:date]. Also supports string formats.'),
      '#default_value' => $this->configuration['time_min'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['time_max'] = [
      '#type' => 'textfield',
      '#title' => $this->t('End Time (Until)'),
      '#description' => $this->t('Upper bound for free/busy query. For date range fields use [entity:field_date_range:end_date], for Smart Date fields use [entity:field_smart_date:end_value], for regular datetime fields use [entity:field_date:date]. Also supports string formats.'),
      '#default_value' => $this->configuration['time_max'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['calendar_ids'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Calendar IDs'),
      '#description' => $this->t('Calendar IDs to check, one per line. Use "primary" for the user\'s primary calendar. Can include tokens like [list_calendars:0:calendar_id].'),
      '#default_value' => $this->configuration['calendar_ids'],
      '#required' => TRUE,
      '#rows' => 5,
      '#eca_token_replacement' => TRUE,
    ];

    $form['freebusy_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Free/Busy Token Name'),
      '#description' => $this->t('Token name to store free/busy results. Accessible as [token_name:calendars:CALENDAR_ID:busy:0:start], [token_name:time_min], etc.'),
      '#default_value' => $this->configuration['freebusy_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $time_min = $this->tokenService->replacePlain($this->configuration['time_min']);
    $time_max = $this->tokenService->replacePlain($this->configuration['time_max']);
    $calendar_ids_text = $this->tokenService->replacePlain($this->configuration['calendar_ids']);
    $freebusy_token_name = $this->tokenService->replacePlain($this->configuration['freebusy_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('check free/busy');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($time_min) || empty($time_max)) {
      $this->logger->error('Google Calendar check free/busy action: Missing time_min or time_max.');
      return;
    }

    if (empty($calendar_ids_text)) {
      $this->logger->error('Google Calendar check free/busy action: Missing calendar_ids.');
      return;
    }


    // Parse calendar IDs from text
    $calendar_ids = [];
    $calendar_lines = explode("\n", trim($calendar_ids_text));
    foreach ($calendar_lines as $line) {
      $calendar_id = trim($line);
      if (!empty($calendar_id)) {
        $calendar_ids[] = $calendar_id;
      }
    }

    if (empty($calendar_ids)) {
      $this->logger->error('Google Calendar check free/busy action: No valid calendar IDs found.');
      return;
    }

    // Format datetime strings for API
    $formatted_time_min = $this->googleCalendarService->formatDateTimeForApi($time_min);
    $formatted_time_max = $this->googleCalendarService->formatDateTimeForApi($time_max);

    // Validate formatted datetimes
    $datetime_validation = [
      ['name' => 'time_min', 'formatted' => $formatted_time_min, 'original' => $time_min],
      ['name' => 'time_max', 'formatted' => $formatted_time_max, 'original' => $time_max],
    ];
    if (!$this->validateFormattedDateTimes($datetime_validation, 'check free/busy')) {
      return;
    }

    $config = [
      'time_min' => $formatted_time_min,
      'time_max' => $formatted_time_max,
      'items' => $calendar_ids,
    ];

    $result = $this->googleCalendarService->checkFreeBusy($auth_type, $client_id, $config);

    if (!$result) {
      $this->logger->error('Google Calendar check free/busy action: Failed to check free/busy status using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store free/busy token data
    if (!empty($freebusy_token_name)) {
      $this->tokenService->addTokenData($freebusy_token_name, $result);
    }

    $calendar_count = count($result['calendars']);
  }

}
