<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Create a new Google Calendar.
 */
#[Action(
  id: 'eca_google_calendar_create_calendar',
  label: new TranslatableMarkup('Google Calendar: Create Calendar'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Creates a new secondary calendar with configurable metadata including name, description, location, and time zone settings.'),
  version_introduced: '1.0.0',
)]
final class CreateCalendar extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'summary' => '',
      'description' => '',
      'location' => '',
      'time_zone' => '',
      'calendar_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['summary'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar Name'),
      '#description' => $this->t('The title/name of the calendar to create.'),
      '#default_value' => $this->configuration['summary'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Calendar Description'),
      '#description' => $this->t('Optional description of the calendar\'s purpose or content.'),
      '#default_value' => $this->configuration['description'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['location'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Location'),
      '#description' => $this->t('Geographic location of the calendar as free-form text.'),
      '#default_value' => $this->configuration['location'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['time_zone'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Time Zone'),
      '#description' => $this->t('Time zone for the calendar (e.g., "America/New_York", "Europe/London"). Leave empty to use system default.'),
      '#default_value' => $this->configuration['time_zone'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['calendar_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar Token Name'),
      '#description' => $this->t('Token name to store the created calendar details. Accessible as [token_name:calendar_id], [token_name:summary], etc.'),
      '#default_value' => $this->configuration['calendar_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $summary = $this->tokenService->replacePlain($this->configuration['summary']);
    $description = $this->tokenService->replacePlain($this->configuration['description'], [], ['clear' => TRUE]);
    $location = $this->tokenService->replacePlain($this->configuration['location'], [], ['clear' => TRUE]);
    $time_zone = $this->tokenService->replacePlain($this->configuration['time_zone'], [], ['clear' => TRUE]);
    $calendar_token_name = $this->tokenService->replacePlain($this->configuration['calendar_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('create calendar');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($summary)) {
      $this->logger->error('Google Calendar create calendar action: Missing calendar name/summary.');
      return;
    }


    $config = [
      'summary' => $summary,
    ];

    if (!empty($description)) {
      $config['description'] = $description;
    }

    if (!empty($location)) {
      $config['location'] = $location;
    }

    if (!empty($time_zone)) {
      $config['time_zone'] = $time_zone;
    }

    $calendar_data = $this->googleCalendarService->createCalendar($auth_type, $client_id, $config);

    if (!$calendar_data) {
      $this->logger->error('Google Calendar create calendar action: Failed to create calendar using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    if (!empty($calendar_token_name)) {
      $this->tokenService->addTokenData($calendar_token_name, $calendar_data);
    }
  }

}
