<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_google_calendar\GoogleCalendarService;

/**
 * Create a new Google Calendar event.
 */
#[Action(
  id: 'eca_google_calendar_create_event',
  label: new TranslatableMarkup('Google Calendar: Create Event'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Creates a new Google Calendar event with full configuration support including attendees, recurring patterns, and visibility settings.'),
  version_introduced: '1.0.0',
)]
final class CreateEvent extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'calendar_id' => GoogleCalendarService::CALENDAR_PRIMARY,
      'summary' => '',
      'description' => '',
      'location' => '',
      'start_time' => '',
      'end_time' => '',
      'timezone' => '',
      'attendees' => '',
      'visibility' => GoogleCalendarService::VISIBILITY_DEFAULT,
      'status' => GoogleCalendarService::STATUS_CONFIRMED,
      'reminders' => '',
      'use_meet' => '',
      'conference_url' => '',
      'event_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);
    $form = $this->addCalendarConfigurationForm($form, $form_state);

    $form['summary'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Event Title'),
      '#description' => $this->t('The title/summary of the event.'),
      '#default_value' => $this->configuration['summary'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Event Description'),
      '#description' => $this->t('Event description. Supports HTML formatting.'),
      '#default_value' => $this->configuration['description'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['location'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Location'),
      '#description' => $this->t('Event location (address, room name, etc).'),
      '#default_value' => $this->configuration['location'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['start_time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Start Time'),
      '#description' => $this->t('Event start time. For date range fields use [entity:field_date_range:start_date], for Smart Date fields use [entity:field_smart_date:value], for regular datetime fields use [entity:field_date:date]. Also supports string formats ("2024-12-25 10:00", "December 25, 2024 10:00 AM").'),
      '#default_value' => $this->configuration['start_time'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['end_time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('End Time'),
      '#description' => $this->t('Event end time. For date range fields use [entity:field_date_range:end_date], for Smart Date fields use [entity:field_smart_date:end_value], for regular datetime fields use [entity:field_date:date]. Also supports string formats ("2024-12-25 11:00", "December 25, 2024 11:00 AM").'),
      '#default_value' => $this->configuration['end_time'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['timezone'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Timezone'),
      '#description' => $this->t('Timezone for the event (e.g., "America/New_York"). Leave empty to use calendar default.'),
      '#default_value' => $this->configuration['timezone'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['attendees'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Attendees'),
      '#description' => $this->t('Email addresses of attendees, one per line.'),
      '#default_value' => $this->configuration['attendees'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['visibility'] = [
      '#type' => 'select',
      '#title' => $this->t('Visibility'),
      '#description' => $this->t('Event visibility setting.'),
      '#options' => $this->googleCalendarService->getVisibilityOptions(),
      '#default_value' => $this->configuration['visibility'],
      '#required' => TRUE,
    ];

    $form['status'] = [
      '#type' => 'select',
      '#title' => $this->t('Status'),
      '#description' => $this->t('Event status.'),
      '#options' => $this->googleCalendarService->getEventStatusOptions(),
      '#default_value' => $this->configuration['status'],
      '#required' => TRUE,
    ];

    $form['reminders'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Reminders'),
      '#description' => $this->t('Event reminders, one per line in format "METHOD MINUTES". Examples:<br/>popup 10<br/>email 1440<br/>Leave empty to use calendar defaults.'),
      '#default_value' => $this->configuration['reminders'],
      '#rows' => 3,
      '#eca_token_replacement' => TRUE,
    ];

    $form['use_meet'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Use Google Meet'),
      '#description' => $this->t('Set to "1", "true", or "yes" to create a Google Meet video conference for this event. Use tokens like [entity:field_checkbox:value] for dynamic control.'),
      '#default_value' => $this->configuration['use_meet'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['conference_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Conference URL'),
      '#description' => $this->t('External video conference URL (e.g., Zoom, Teams, custom meeting link). Use this for non-Google meeting platforms or pre-created meetings.'),
      '#default_value' => $this->configuration['conference_url'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['event_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Event Token Name'),
      '#description' => $this->t('Optional token name to store event details. Accessible as [token_name:event_id], [token_name:event_url], [token_name:status].'),
      '#default_value' => $this->configuration['event_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $calendar_id = $this->tokenService->replacePlain($this->configuration['calendar_id']);
    $summary = $this->tokenService->replacePlain($this->configuration['summary']);
    $description = $this->tokenService->replacePlain($this->configuration['description'], [], ['clear' => TRUE]);
    $location = $this->tokenService->replacePlain($this->configuration['location'], [], ['clear' => TRUE]);
    $start_time = $this->tokenService->replacePlain($this->configuration['start_time']);
    $end_time = $this->tokenService->replacePlain($this->configuration['end_time']);
    $timezone = $this->tokenService->replacePlain($this->configuration['timezone'], [], ['clear' => TRUE]);
    $attendees_text = $this->tokenService->replacePlain($this->configuration['attendees'], [], ['clear' => TRUE]);
    $visibility = $this->configuration['visibility'];
    $status = $this->configuration['status'];
    $reminders_text = $this->tokenService->replacePlain($this->configuration['reminders'], [], ['clear' => TRUE]);
    $use_meet_text = $this->tokenService->replacePlain($this->configuration['use_meet']);
    $use_meet = !empty($use_meet_text);
    $conference_url = $this->tokenService->replacePlain($this->configuration['conference_url'], [], ['clear' => TRUE]);
    $event_token_name = $this->tokenService->replacePlain($this->configuration['event_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('create event');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($summary)) {
      $this->logger->error('Google Calendar create event action: Missing event summary/title.');
      return;
    }

    if (empty($start_time) || empty($end_time)) {
      $this->logger->error('Google Calendar create event action: Missing start_time or end_time.');
      return;
    }


    // Parse attendees
    $attendees = $this->googleCalendarService->parseAttendees($attendees_text);

    // Parse reminders
    $reminders = $this->googleCalendarService->parseReminders($reminders_text);

    // Check if this is an all-day event (both start and end must qualify)
    $is_all_day = $this->googleCalendarService->isAllDayEventRange($start_time, $end_time, $timezone);

    // Format datetime strings - force consistent formatting based on all-day status
    if ($is_all_day) {
      // For all-day events, force date-only formatting
      $formatted_start_time = $this->googleCalendarService->formatDateForAllDay($start_time, $timezone);
      $formatted_end_time = $this->googleCalendarService->formatDateForAllDay($end_time, $timezone);
    } else {
      // For timed events, force datetime formatting
      $formatted_start_time = $this->googleCalendarService->formatDateTimeForApi($start_time, $timezone, FALSE);
      $formatted_end_time = $this->googleCalendarService->formatDateTimeForApi($end_time, $timezone, FALSE);
    }
    // Validate formatted datetimes
    $datetime_validation = [
      ['name' => 'start_time', 'formatted' => $formatted_start_time, 'original' => $start_time],
      ['name' => 'end_time', 'formatted' => $formatted_end_time, 'original' => $end_time],
    ];
    if (!$this->validateFormattedDateTimes($datetime_validation, 'create event')) {
      return;
    }

    $config = [
      'calendar_id' => $calendar_id ?: GoogleCalendarService::CALENDAR_PRIMARY,
      'summary' => $summary,
      'description' => $description,
      'location' => $location,
      'start_time' => $formatted_start_time,
      'end_time' => $formatted_end_time,
      'start_time_original' => $start_time,
      'end_time_original' => $end_time,
      'timezone' => $timezone,
      'attendees' => $attendees,
      'visibility' => $visibility,
      'status' => $status,
      'reminders' => $reminders,
      'use_meet' => $use_meet,
      'conference_url' => $conference_url,
    ];

    $event_data = $this->googleCalendarService->createEvent($auth_type, $client_id, $config);

    if (!$event_data) {
      $this->logger->error('Google Calendar create event action: Failed to create event using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    if (!empty($event_token_name)) {
      $this->tokenService->addTokenData($event_token_name, $event_data);
    }
  }

}
