<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Delete a Google Calendar.
 */
#[Action(
  id: 'eca_google_calendar_delete_calendar',
  label: new TranslatableMarkup('Google Calendar: Delete Calendar'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Deletes a secondary Google Calendar. Cannot delete primary calendars - use calendar clearing instead. This action is permanent and cannot be undone.'),
  version_introduced: '1.0.0',
)]
final class DeleteCalendar extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'calendar_id' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['calendar_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar ID'),
      '#description' => $this->t('The ID of the secondary calendar to delete. Cannot use "primary" - primary calendars cannot be deleted. <strong>Warning:</strong> This action is permanent and cannot be undone.'),
      '#default_value' => $this->configuration['calendar_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $calendar_id = $this->tokenService->replacePlain($this->configuration['calendar_id']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('delete calendar');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($calendar_id)) {
      $this->logger->error('Google Calendar delete calendar action: Missing calendar_id.');
      return;
    }

    // Additional validation for primary calendar
    if ($calendar_id === 'primary') {
      $this->logger->error('Google Calendar delete calendar action: Cannot delete primary calendar. Use calendar clearing instead.');
      return;
    }


    $success = $this->googleCalendarService->deleteCalendar($auth_type, $client_id, $calendar_id);

    if (!$success) {
      $this->logger->error('Google Calendar delete calendar action: Failed to delete calendar @calendar_id using @auth_type.', [
        '@calendar_id' => $calendar_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }
  }

}
