<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_google_calendar\GoogleCalendarService;

/**
 * Delete a Google Calendar event.
 */
#[Action(
  id: 'eca_google_calendar_delete_event',
  label: new TranslatableMarkup('Google Calendar: Delete Event'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Deletes a Google Calendar event and sends cancellation notifications to attendees.'),
  version_introduced: '1.0.0',
)]
final class DeleteEvent extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'calendar_id' => GoogleCalendarService::CALENDAR_PRIMARY,
      'event_id' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);
    $form = $this->addCalendarConfigurationForm($form, $form_state);

    $form['event_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Event ID'),
      '#description' => $this->t('The ID of the event to delete.'),
      '#default_value' => $this->configuration['event_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $calendar_id = $this->tokenService->replacePlain($this->configuration['calendar_id']);
    $event_id = $this->tokenService->replacePlain($this->configuration['event_id']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('delete event');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($event_id)) {
      $this->logger->error('Google Calendar delete event action: Missing event_id.');
      return;
    }


    $config = [
      'calendar_id' => $calendar_id ?: GoogleCalendarService::CALENDAR_PRIMARY,
    ];

    $success = $this->googleCalendarService->deleteEvent($auth_type, $client_id, $event_id, $config);

    if (!$success) {
      $this->logger->error('Google Calendar delete event action: Failed to delete event @event_id using @auth_type.', [
        '@event_id' => $event_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }
  }

}
