<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_google_calendar\GoogleCalendarService;

/**
 * Get detailed information for a specific Google Calendar event.
 */
#[Action(
  id: 'eca_google_calendar_get_event',
  label: new TranslatableMarkup('Google Calendar: Get Event'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieves detailed information for a specific Google Calendar event including attendee responses and full metadata.'),
  version_introduced: '1.0.0',
)]
final class GetEvent extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'calendar_id' => GoogleCalendarService::CALENDAR_PRIMARY,
      'event_id' => '',
      'event_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);
    $form = $this->addCalendarConfigurationForm($form, $form_state);

    $form['event_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Event ID'),
      '#description' => $this->t('The ID of the event to retrieve.'),
      '#default_value' => $this->configuration['event_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['event_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Event Token Name'),
      '#description' => $this->t('Token name to store event details. Accessible as [token_name:event_id], [token_name:summary], [token_name:attendees:0:email], etc.'),
      '#default_value' => $this->configuration['event_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $calendar_id = $this->tokenService->replacePlain($this->configuration['calendar_id']);
    $event_id = $this->tokenService->replacePlain($this->configuration['event_id']);
    $event_token_name = $this->tokenService->replacePlain($this->configuration['event_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('get event');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($event_id)) {
      $this->logger->error('Google Calendar get event action: Missing event_id.');
      return;
    }


    $config = [
      'calendar_id' => $calendar_id ?: GoogleCalendarService::CALENDAR_PRIMARY,
    ];

    $event_data = $this->googleCalendarService->getEvent($auth_type, $client_id, $event_id, $config);

    if (!$event_data) {
      $this->logger->error('Google Calendar get event action: Failed to retrieve event @event_id using @auth_type.', [
        '@event_id' => $event_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    if (!empty($event_token_name)) {
      $this->tokenService->addTokenData($event_token_name, $event_data);
    }

  }

}
