<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * List access permissions for a Google Calendar.
 */
#[Action(
  id: 'eca_google_calendar_list_access',
  label: new TranslatableMarkup('Google Calendar: List Access'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieves all access control rules for a calendar, including user permissions, roles, and ACL rule IDs that can be used with other access management actions.'),
  version_introduced: '1.0.0',
)]
final class ListCalendarAccess extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'calendar_id' => '',
      'access_rules_token_name' => '',
      'count_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['calendar_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar ID'),
      '#description' => $this->t('The ID of the calendar to list access rules for. Use "primary" for the user\'s primary calendar.'),
      '#default_value' => $this->configuration['calendar_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['access_rules_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Access Rules Token Name'),
      '#description' => $this->t('Token name to store the array of access rules. Accessible as [token_name:0:acl_id], [token_name:0:role], [token_name:0:scope_type], [token_name:0:scope_value], etc.'),
      '#default_value' => $this->configuration['access_rules_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['count_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Count Token Name'),
      '#description' => $this->t('Token name to store the total number of access rules found.'),
      '#default_value' => $this->configuration['count_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $calendar_id = $this->tokenService->replacePlain($this->configuration['calendar_id']);
    $access_rules_token_name = $this->tokenService->replacePlain($this->configuration['access_rules_token_name']);
    $count_token_name = $this->tokenService->replacePlain($this->configuration['count_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('list calendar access');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($calendar_id)) {
      $this->logger->error('Google Calendar list access action: Missing calendar_id.');
      return;
    }


    $config = [
      'operation' => 'list',
    ];

    $result = $this->googleCalendarService->manageCalendarAccess($auth_type, $client_id, $calendar_id, $config);

    if (!$result) {
      $this->logger->error('Google Calendar list access action: Failed to list access rules for calendar @calendar_id using @auth_type.', [
        '@calendar_id' => $calendar_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store access rules using standard numeric indexing
    if (!empty($access_rules_token_name)) {
      $this->tokenService->addTokenData($access_rules_token_name, $result['acl_rules']);
    }

    // Store count
    if (!empty($count_token_name)) {
      $this->tokenService->addTokenData($count_token_name, ['count' => $result['count']]);
    }
  }

}
