<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * List Google Calendars accessible to the authenticated user.
 */
#[Action(
  id: 'eca_google_calendar_list_calendars',
  label: new TranslatableMarkup('Google Calendar: List Calendars'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieves all calendars accessible to the authenticated user with filtering options for access roles, visibility, and pagination.'),
  version_introduced: '1.0.0',
)]
final class ListCalendars extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'max_results' => '50',
      'min_access_role' => '',
      'show_deleted' => FALSE,
      'show_hidden' => FALSE,
      'calendars_token_name' => '',
      'count_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['max_results'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Results'),
      '#description' => $this->t('Maximum number of calendars to return (1-250).'),
      '#default_value' => $this->configuration['max_results'],
      '#min' => 1,
      '#max' => 250,
      '#required' => TRUE,
    ];

    $access_role_options = ['' => $this->t('- No minimum role filter -')] + $this->googleCalendarService->getAccessRoleOptions();
    $form['min_access_role'] = [
      '#type' => 'select',
      '#title' => $this->t('Minimum Access Role'),
      '#description' => $this->t('Filter calendars by minimum access role. Only calendars with this role or higher will be returned.'),
      '#options' => $access_role_options,
      '#default_value' => $this->configuration['min_access_role'],
    ];

    $form['show_deleted'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include Deleted Calendars'),
      '#description' => $this->t('Whether to include deleted calendar entries in the result.'),
      '#default_value' => $this->configuration['show_deleted'],
    ];

    $form['show_hidden'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include Hidden Calendars'),
      '#description' => $this->t('Whether to include hidden calendar entries in the result.'),
      '#default_value' => $this->configuration['show_hidden'],
    ];

    $form['calendars_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendars Token Name'),
      '#description' => $this->t('Token name to store the array of calendars. Accessible as [token_name:0:calendar_id], [token_name:1:summary], etc.'),
      '#default_value' => $this->configuration['calendars_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['count_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Count Token Name'),
      '#description' => $this->t('Token name to store the total number of calendars found.'),
      '#default_value' => $this->configuration['count_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $max_results = $this->configuration['max_results'];
    $min_access_role = $this->configuration['min_access_role'];
    $show_deleted = $this->configuration['show_deleted'];
    $show_hidden = $this->configuration['show_hidden'];
    $calendars_token_name = $this->tokenService->replacePlain($this->configuration['calendars_token_name']);
    $count_token_name = $this->tokenService->replacePlain($this->configuration['count_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('list calendars');
    if (!$auth_type || !$client_id) {
      return;
    }


    $config = [
      'max_results' => $max_results,
      'show_deleted' => $show_deleted,
      'show_hidden' => $show_hidden,
    ];

    if (!empty($min_access_role)) {
      $config['min_access_role'] = $min_access_role;
    }

    $result = $this->googleCalendarService->listCalendars($auth_type, $client_id, $config);

    if (!$result) {
      $this->logger->error('Google Calendar list calendars action: Failed to retrieve calendars using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store calendars token data using standard numeric indexing
    if (!empty($calendars_token_name)) {
      $this->tokenService->addTokenData($calendars_token_name, $result['calendars']);
    }

    // Store count
    if (!empty($count_token_name)) {
      $this->tokenService->addTokenData($count_token_name, ['count' => $result['count']]);
    }
  }

}
