<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_google_calendar\GoogleCalendarService;

/**
 * List Google Calendar events with filtering.
 */
#[Action(
  id: 'eca_google_calendar_list_events',
  label: new TranslatableMarkup('Google Calendar: List Events'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Lists Google Calendar events with advanced filtering options including date ranges, text search, and status filtering.'),
  version_introduced: '1.0.0',
)]
final class ListEvents extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'calendar_id' => GoogleCalendarService::CALENDAR_PRIMARY,
      'time_min' => '',
      'time_max' => '',
      'search_query' => '',
      'max_results' => '50',
      'order_by' => GoogleCalendarService::ORDER_BY_START_TIME,
      'single_events' => TRUE,
      'events_token_name' => '',
      'count_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);
    $form = $this->addCalendarConfigurationForm($form, $form_state);

    $form['time_min'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Start Time (From)'),
      '#description' => $this->t('Lower bound for events start time. For date range fields use [entity:field_date_range:start_date], for Smart Date fields use [entity:field_smart_date:value], for regular datetime fields use [entity:field_date:date]. Also supports string formats. Leave empty for no lower bound.'),
      '#default_value' => $this->configuration['time_min'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['time_max'] = [
      '#type' => 'textfield',
      '#title' => $this->t('End Time (Until)'),
      '#description' => $this->t('Upper bound for events start time. For date range fields use [entity:field_date_range:start_date], for Smart Date fields use [entity:field_smart_date:value], for regular datetime fields use [entity:field_date:date]. Also supports string formats. Leave empty for no upper bound.'),
      '#default_value' => $this->configuration['time_max'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['search_query'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search Query'),
      '#description' => $this->t('Free text search terms to find events whose summaries, descriptions, locations, attendees names or emails contain these terms.'),
      '#default_value' => $this->configuration['search_query'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['max_results'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Results'),
      '#description' => $this->t('Maximum number of events to return. Maximum value is 2500.'),
      '#default_value' => $this->configuration['max_results'],
      '#min' => 1,
      '#max' => 2500,
      '#required' => TRUE,
    ];

    $form['order_by'] = [
      '#type' => 'select',
      '#title' => $this->t('Order By'),
      '#description' => $this->t('The order of the events returned in the result.'),
      '#options' => [
        GoogleCalendarService::ORDER_BY_START_TIME => $this->t('Start Time'),
        GoogleCalendarService::ORDER_BY_UPDATED => $this->t('Last Modified Time'),
      ],
      '#default_value' => $this->configuration['order_by'],
      '#required' => TRUE,
    ];

    $form['single_events'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Expand Recurring Events'),
      '#description' => $this->t('Whether to expand recurring events into instances and only return single one-off events and instances of recurring events.'),
      '#default_value' => $this->configuration['single_events'],
    ];

    $form['events_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Events Token Name'),
      '#description' => $this->t('Token name to store the array of events. Accessible as [token_name:0:event_id], [token_name:1:summary], etc.'),
      '#default_value' => $this->configuration['events_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['count_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Count Token Name'),
      '#description' => $this->t('Token name to store the total number of events found.'),
      '#default_value' => $this->configuration['count_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $calendar_id = $this->tokenService->replacePlain($this->configuration['calendar_id']);
    $time_min = $this->tokenService->replacePlain($this->configuration['time_min']);
    $time_max = $this->tokenService->replacePlain($this->configuration['time_max']);
    $search_query = $this->tokenService->replacePlain($this->configuration['search_query']);
    $max_results = $this->configuration['max_results'];
    $order_by = $this->configuration['order_by'];
    $single_events = $this->configuration['single_events'];
    $events_token_name = $this->tokenService->replacePlain($this->configuration['events_token_name']);
    $count_token_name = $this->tokenService->replacePlain($this->configuration['count_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('list events');
    if (!$auth_type || !$client_id) {
      return;
    }


    $config = [
      'calendar_id' => $calendar_id ?: GoogleCalendarService::CALENDAR_PRIMARY,
      'max_results' => $max_results,
      'order_by' => $order_by,
      'single_events' => $single_events,
    ];

    if (!empty($time_min)) {
      $formatted_time_min = $this->googleCalendarService->formatDateTimeForApi($time_min);
      if ($formatted_time_min) {
        $config['time_min'] = $formatted_time_min;
      }
    }

    if (!empty($time_max)) {
      $formatted_time_max = $this->googleCalendarService->formatDateTimeForApi($time_max);
      if ($formatted_time_max) {
        $config['time_max'] = $formatted_time_max;
      }
    }

    if (!empty($search_query)) {
      $config['q'] = $search_query;
    }

    $result = $this->googleCalendarService->listEvents($auth_type, $client_id, $config);

    if (!$result) {
      $this->logger->error('Google Calendar list events action: Failed to retrieve events using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store events token data using standard numeric indexing
    if (!empty($events_token_name)) {
      $this->tokenService->addTokenData($events_token_name, $result['events']);
    }

    // Store count
    if (!empty($count_token_name)) {
      $this->tokenService->addTokenData($count_token_name, ['count' => $result['count']]);
    }
  }

}
