<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Remove access permissions from a Google Calendar.
 */
#[Action(
  id: 'eca_google_calendar_remove_access',
  label: new TranslatableMarkup('Google Calendar: Remove Access'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Revokes calendar access permissions by removing an existing ACL rule. Use the List Access action to find ACL rule IDs.'),
  version_introduced: '1.0.0',
)]
final class RemoveCalendarAccess extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'calendar_id' => '',
      'acl_id' => '',
      'success_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['calendar_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar ID'),
      '#description' => $this->t('The ID of the calendar to remove access from. Use "primary" for the user\'s primary calendar.'),
      '#default_value' => $this->configuration['calendar_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['acl_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('ACL Rule ID'),
      '#description' => $this->t('The ID of the access control rule to remove. Use the "List Access" action to find ACL rule IDs.'),
      '#default_value' => $this->configuration['acl_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['success_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Success Token Name'),
      '#description' => $this->t('Token name to store the removal result. Accessible as [token_name:success], [token_name:acl_id].'),
      '#default_value' => $this->configuration['success_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['help'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--info"><h4>' . $this->t('Finding ACL Rule IDs') . '</h4><p>' . $this->t('Use the "Google Calendar: List Access" action to retrieve all access rules for a calendar. Each rule will have an ACL ID that you can use with this action to remove specific access permissions.') . '</p></div>',
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $calendar_id = $this->tokenService->replacePlain($this->configuration['calendar_id']);
    $acl_id = $this->tokenService->replacePlain($this->configuration['acl_id']);
    $success_token_name = $this->tokenService->replacePlain($this->configuration['success_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('remove calendar access');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($calendar_id)) {
      $this->logger->error('Google Calendar remove access action: Missing calendar_id.');
      return;
    }

    if (empty($acl_id)) {
      $this->logger->error('Google Calendar remove access action: Missing acl_id.');
      return;
    }


    $config = [
      'operation' => 'remove',
      'acl_id' => $acl_id,
    ];

    $result = $this->googleCalendarService->manageCalendarAccess($auth_type, $client_id, $calendar_id, $config);

    if (!$result) {
      $this->logger->error('Google Calendar remove access action: Failed to remove access rule @acl_id from calendar @calendar_id using @auth_type.', [
        '@acl_id' => $acl_id,
        '@calendar_id' => $calendar_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store success result
    if (!empty($success_token_name)) {
      $this->tokenService->addTokenData($success_token_name, $result);
    }
  }

}
