<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Update an existing Google Calendar.
 */
#[Action(
  id: 'eca_google_calendar_update_calendar',
  label: new TranslatableMarkup('Google Calendar: Update Calendar'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Updates metadata for an existing calendar including name, description, location, and time zone settings. Only provided fields will be modified.'),
  version_introduced: '1.0.0',
)]
final class UpdateCalendar extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'calendar_id' => '',
      'summary' => '',
      'description' => '',
      'location' => '',
      'time_zone' => '',
      'calendar_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['calendar_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar ID'),
      '#description' => $this->t('The ID of the calendar to update. Use "primary" for the user\'s primary calendar.'),
      '#default_value' => $this->configuration['calendar_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['summary'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar Name'),
      '#description' => $this->t('The new title/name for the calendar. Leave empty to keep current value.'),
      '#default_value' => $this->configuration['summary'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Calendar Description'),
      '#description' => $this->t('New description for the calendar. Leave empty to keep current value.'),
      '#default_value' => $this->configuration['description'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['location'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Location'),
      '#description' => $this->t('Geographic location of the calendar. Leave empty to keep current value.'),
      '#default_value' => $this->configuration['location'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['time_zone'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Time Zone'),
      '#description' => $this->t('Time zone for the calendar (e.g., "America/New_York", "Europe/London"). Leave empty to keep current value.'),
      '#default_value' => $this->configuration['time_zone'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['calendar_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar Token Name'),
      '#description' => $this->t('Optional token name to store the updated calendar details. Accessible as [token_name:calendar_id], [token_name:summary], etc.'),
      '#default_value' => $this->configuration['calendar_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $calendar_id = $this->tokenService->replacePlain($this->configuration['calendar_id']);
    $summary = $this->tokenService->replacePlain($this->configuration['summary']);
    $description = $this->tokenService->replacePlain($this->configuration['description'], [], ['clear' => TRUE]);
    $location = $this->tokenService->replacePlain($this->configuration['location'], [], ['clear' => TRUE]);
    $time_zone = $this->tokenService->replacePlain($this->configuration['time_zone'], [], ['clear' => TRUE]);
    $calendar_token_name = $this->tokenService->replacePlain($this->configuration['calendar_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('update calendar');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($calendar_id)) {
      $this->logger->error('Google Calendar update calendar action: Missing calendar_id.');
      return;
    }


    $config = [];

    // Include values based on user intent: if user provided input (non-empty config), 
    // always pass it even if token evaluates to empty (user wants to clear field)
    if (!empty($this->configuration['summary'])) {
      $config['summary'] = $summary;
    }

    if (!empty($this->configuration['description'])) {
      $config['description'] = $description;
    }

    if (!empty($this->configuration['location'])) {
      $config['location'] = $location;
    }

    if (!empty($this->configuration['time_zone'])) {
      $config['time_zone'] = $time_zone;
    }

    // Check if we have any fields to update
    if (empty($config)) {
      $this->logger->warning('Google Calendar update calendar action: No fields provided for update.');
      return;
    }

    $calendar_data = $this->googleCalendarService->updateCalendar($auth_type, $client_id, $calendar_id, $config);

    if (!$calendar_data) {
      $this->logger->error('Google Calendar update calendar action: Failed to update calendar @calendar_id using @auth_type.', [
        '@calendar_id' => $calendar_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    if (!empty($calendar_token_name)) {
      $this->tokenService->addTokenData($calendar_token_name, $calendar_data);
    }
  }

}
