<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_google_calendar\GoogleCalendarService;

/**
 * Update access permissions for a Google Calendar.
 */
#[Action(
  id: 'eca_google_calendar_update_access',
  label: new TranslatableMarkup('Google Calendar: Update Access'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Modifies existing calendar access permissions by updating roles or scope information. Use the List Access action to find ACL rule IDs.'),
  version_introduced: '1.0.0',
)]
final class UpdateCalendarAccess extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'calendar_id' => '',
      'acl_id' => '',
      'role' => '',
      'scope_type' => '',
      'scope_value' => '',
      'send_notifications' => FALSE,
      'access_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['calendar_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar ID'),
      '#description' => $this->t('The ID of the calendar to update access for. Use "primary" for the user\'s primary calendar.'),
      '#default_value' => $this->configuration['calendar_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['acl_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('ACL Rule ID'),
      '#description' => $this->t('The ID of the access control rule to update. Use the "List Access" action to find ACL rule IDs.'),
      '#default_value' => $this->configuration['acl_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $access_role_options = ['' => $this->t('- Keep current role -')] + $this->googleCalendarService->getAccessRoleOptions();
    $form['role'] = [
      '#type' => 'select',
      '#title' => $this->t('Access Role'),
      '#description' => $this->t('The new level of access to assign. Leave as "Keep current role" to maintain the existing role.'),
      '#options' => $access_role_options,
      '#default_value' => $this->configuration['role'],
    ];

    $scope_type_options = ['' => $this->t('- Keep current scope type -')] + $this->googleCalendarService->getAclScopeTypeOptions();
    $form['scope_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Access Scope Type'),
      '#description' => $this->t('The new type of entity for access. Leave as "Keep current scope type" to maintain the existing scope type.'),
      '#options' => $scope_type_options,
      '#default_value' => $this->configuration['scope_type'],
    ];

    $form['scope_value'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Email/Domain'),
      '#description' => $this->t('New email address for user/group access, domain name for domain access, or leave empty to keep current value.'),
      '#default_value' => $this->configuration['scope_value'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['send_notifications'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send Notifications'),
      '#description' => $this->t('Whether to send email notifications about the access change.'),
      '#default_value' => $this->configuration['send_notifications'],
    ];

    $form['access_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Access Token Name'),
      '#description' => $this->t('Token name to store the updated access rule details. Accessible as [token_name:acl_id], [token_name:role], [token_name:scope_type], [token_name:scope_value].'),
      '#default_value' => $this->configuration['access_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['help'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--info"><h4>' . $this->t('Finding ACL Rule IDs') . '</h4><p>' . $this->t('Use the "Google Calendar: List Access" action to retrieve all access rules for a calendar. Each rule will have an ACL ID that you can use with this action to update specific access permissions. Only provide the fields you want to change - leave others empty to keep existing values.') . '</p></div>',
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $calendar_id = $this->tokenService->replacePlain($this->configuration['calendar_id']);
    $acl_id = $this->tokenService->replacePlain($this->configuration['acl_id']);
    $role = $this->configuration['role'];
    $scope_type = $this->configuration['scope_type'];
    $scope_value = $this->tokenService->replacePlain($this->configuration['scope_value'], [], ['clear' => TRUE]);
    $send_notifications = $this->configuration['send_notifications'];
    $access_token_name = $this->tokenService->replacePlain($this->configuration['access_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('update calendar access');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($calendar_id)) {
      $this->logger->error('Google Calendar update access action: Missing calendar_id.');
      return;
    }

    if (empty($acl_id)) {
      $this->logger->error('Google Calendar update access action: Missing acl_id.');
      return;
    }

    // Check if at least one field is provided for update
    if (empty($role) && empty($scope_type) && empty($scope_value)) {
      $this->logger->warning('Google Calendar update access action: No update fields provided.');
      return;
    }


    $config = [
      'operation' => 'update',
      'acl_id' => $acl_id,
      'role' => $role,
      'scope_type' => $scope_type,
      'scope_value' => $scope_value,
      'send_notifications' => $send_notifications,
    ];

    $result = $this->googleCalendarService->manageCalendarAccess($auth_type, $client_id, $calendar_id, $config);

    if (!$result) {
      $this->logger->error('Google Calendar update access action: Failed to update access rule @acl_id for calendar @calendar_id using @auth_type.', [
        '@acl_id' => $acl_id,
        '@calendar_id' => $calendar_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store updated access rule data
    if (!empty($access_token_name)) {
      $this->tokenService->addTokenData($access_token_name, $result);
    }

  }

}
