<?php

declare(strict_types=1);

namespace Drupal\eca_google_calendar\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_google_calendar\GoogleCalendarService;

/**
 * Update an existing Google Calendar event.
 */
#[Action(
  id: 'eca_google_calendar_update_event',
  label: new TranslatableMarkup('Google Calendar: Update Event'),
  category: new TranslatableMarkup('Google Calendar'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Updates an existing Google Calendar event with new properties and optionally sends notifications to attendees.'),
  version_introduced: '1.0.0',
)]
final class UpdateEvent extends CalendarActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'calendar_id' => GoogleCalendarService::CALENDAR_PRIMARY,
      'event_id' => '',
      'summary' => '',
      'description' => '',
      'location' => '',
      'start_time' => '',
      'end_time' => '',
      'timezone' => '',
      'attendees' => '',
      'visibility' => '',
      'status' => '',
      'reminders' => '',
      'use_meet' => '',
      'conference_url' => '',
      'event_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);
    $form = $this->addCalendarConfigurationForm($form, $form_state);

    $form['event_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Event ID'),
      '#description' => $this->t('The ID of the event to update.'),
      '#default_value' => $this->configuration['event_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['summary'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Event Title'),
      '#description' => $this->t('The title/summary of the event. Leave empty to keep current value.'),
      '#default_value' => $this->configuration['summary'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Event Description'),
      '#description' => $this->t('Event description. Supports HTML formatting. Leave empty to keep current value.'),
      '#default_value' => $this->configuration['description'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['location'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Location'),
      '#description' => $this->t('Event location. Leave empty to keep current value.'),
      '#default_value' => $this->configuration['location'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['start_time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Start Time'),
      '#description' => $this->t('Event start time. For date range fields use [entity:field_date_range:start_date], for Smart Date fields use [entity:field_smart_date:value], for regular datetime fields use [entity:field_date:date]. Leave empty to keep current value.'),
      '#default_value' => $this->configuration['start_time'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['end_time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('End Time'),
      '#description' => $this->t('Event end time. For date range fields use [entity:field_date_range:end_date], for Smart Date fields use [entity:field_smart_date:end_value], for regular datetime fields use [entity:field_date:date]. Leave empty to keep current value.'),
      '#default_value' => $this->configuration['end_time'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['timezone'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Timezone'),
      '#description' => $this->t('Timezone for the event. Leave empty to keep current value.'),
      '#default_value' => $this->configuration['timezone'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['attendees'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Attendees'),
      '#description' => $this->t('Email addresses of attendees, one per line. Leave empty to keep current attendees.'),
      '#default_value' => $this->configuration['attendees'],
      '#eca_token_replacement' => TRUE,
    ];

    $visibility_options = ['' => $this->t('- Keep current value -')] + $this->googleCalendarService->getVisibilityOptions();
    $form['visibility'] = [
      '#type' => 'select',
      '#title' => $this->t('Visibility'),
      '#description' => $this->t('Event visibility setting.'),
      '#options' => $visibility_options,
      '#default_value' => $this->configuration['visibility'],
    ];

    $status_options = ['' => $this->t('- Keep current value -')] + $this->googleCalendarService->getEventStatusOptions();
    $form['status'] = [
      '#type' => 'select',
      '#title' => $this->t('Status'),
      '#description' => $this->t('Event status.'),
      '#options' => $status_options,
      '#default_value' => $this->configuration['status'],
    ];

    $form['reminders'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Reminders'),
      '#description' => $this->t('Event reminders, one per line in format "METHOD MINUTES". Examples:<br/>popup 10<br/>email 1440<br/>Leave empty to keep current reminders.'),
      '#default_value' => $this->configuration['reminders'],
      '#rows' => 3,
      '#eca_token_replacement' => TRUE,
    ];

    $form['use_meet'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Use Google Meet'),
      '#description' => $this->t('Set to "1", "true", or "yes" to ensure Google Meet exists. Set to "0", "false", or "no" to allow external conference or remove conference. Use tokens like [entity:field_checkbox:value].'),
      '#default_value' => $this->configuration['use_meet'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['conference_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Conference URL'),
      '#description' => $this->t('External video conference URL (e.g., Zoom, Teams, custom meeting link). Leave empty to keep current conference data. This will replace any existing conference data.'),
      '#default_value' => $this->configuration['conference_url'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['event_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Event Token Name'),
      '#description' => $this->t('Optional token name to store updated event details.'),
      '#default_value' => $this->configuration['event_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $calendar_id = $this->tokenService->replacePlain($this->configuration['calendar_id']);
    $event_id = $this->tokenService->replacePlain($this->configuration['event_id']);
    $summary = $this->tokenService->replacePlain($this->configuration['summary']);
    $description = $this->tokenService->replace($this->configuration['description'], [], ['clear' => TRUE]);
    $location = $this->tokenService->replacePlain($this->configuration['location'], [], ['clear' => TRUE]);
    $start_time = $this->tokenService->replacePlain($this->configuration['start_time']);
    $end_time = $this->tokenService->replacePlain($this->configuration['end_time']);
    $timezone = $this->tokenService->replacePlain($this->configuration['timezone'], [], ['clear' => TRUE]);
    $attendees_text = $this->tokenService->replacePlain($this->configuration['attendees'], [], ['clear' => TRUE]);
    $visibility = $this->configuration['visibility'];
    $status = $this->configuration['status'];
    $reminders_text = $this->tokenService->replacePlain($this->configuration['reminders'], [], ['clear' => TRUE]);
    $use_meet_text = $this->tokenService->replacePlain($this->configuration['use_meet']);
    $use_meet = !empty($use_meet_text);
    $conference_url = $this->tokenService->replacePlain($this->configuration['conference_url'], [], ['clear' => TRUE]);
    $event_token_name = $this->tokenService->replacePlain($this->configuration['event_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('update event');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($event_id)) {
      $this->logger->error('Google Calendar update event action: Missing event_id.');
      return;
    }


    $config = [
      'calendar_id' => $calendar_id ?: GoogleCalendarService::CALENDAR_PRIMARY,
    ];

    // Include values based on user intent: if user provided input (non-empty config),
    // always pass it even if token evaluates to empty (user wants to clear field)
    if (!empty($this->configuration['summary'])) {
      $config['summary'] = $summary;
    }

    if (!empty($this->configuration['description'])) {
      $config['description'] = $description;
    }

    if (!empty($this->configuration['location'])) {
      $config['location'] = $location;
    }

    // Handle datetime formatting with consistent all-day logic
    if (!empty($start_time) && !empty($end_time)) {
      // Both times provided - check for all-day consistency
      $is_all_day = $this->googleCalendarService->isAllDayEventRange($start_time, $end_time, $timezone);

      if ($is_all_day) {
        $formatted_start_time = $this->googleCalendarService->formatDateForAllDay($start_time, $timezone);
        $formatted_end_time = $this->googleCalendarService->formatDateForAllDay($end_time, $timezone);
      } else {
        $formatted_start_time = $this->googleCalendarService->formatDateTimeForApi($start_time, $timezone, FALSE);
        $formatted_end_time = $this->googleCalendarService->formatDateTimeForApi($end_time, $timezone, FALSE);
      }

      // Validate formatted datetimes
      $datetime_validation = [
        ['name' => 'start_time', 'formatted' => $formatted_start_time, 'original' => $start_time],
        ['name' => 'end_time', 'formatted' => $formatted_end_time, 'original' => $end_time],
      ];
      if (!$this->validateFormattedDateTimes($datetime_validation, 'update event')) {
        return;
      }

      $config['start_time'] = $formatted_start_time;
      $config['end_time'] = $formatted_end_time;
      $config['start_time_original'] = $start_time;
      $config['end_time_original'] = $end_time;
    } elseif (!empty($start_time)) {
      // Only start time provided - use auto-detection
      $formatted_start_time = $this->googleCalendarService->formatDateTimeForApi($start_time, $timezone);
      if (!$this->validateFormattedDateTimes([['name' => 'start_time', 'formatted' => $formatted_start_time, 'original' => $start_time]], 'update event')) {
        return;
      }
      $config['start_time'] = $formatted_start_time;
      $config['start_time_original'] = $start_time;
    } elseif (!empty($end_time)) {
      // Only end time provided - use auto-detection
      $formatted_end_time = $this->googleCalendarService->formatDateTimeForApi($end_time, $timezone);
      if (!$this->validateFormattedDateTimes([['name' => 'end_time', 'formatted' => $formatted_end_time, 'original' => $end_time]], 'update event')) {
        return;
      }
      $config['end_time'] = $formatted_end_time;
      $config['end_time_original'] = $end_time;
    }

    if (!empty($timezone)) {
      $config['timezone'] = $timezone;
    }

    // Parse attendees based on user intent
    if (!empty($this->configuration['attendees'])) {
      $config['attendees'] = $this->googleCalendarService->parseAttendees($attendees_text);
    }

    if (!empty($visibility)) {
      $config['visibility'] = $visibility;
    }

    if (!empty($status)) {
      $config['status'] = $status;
    }

    // Parse reminders based on user intent
    if (!empty($this->configuration['reminders'])) {
      $config['reminders'] = $this->googleCalendarService->parseReminders($reminders_text);
    }

    if ($use_meet) {
      $config['use_meet'] = TRUE;
    }

    // Handle conference URL based on user intent
    if (!empty($this->configuration['conference_url'])) {
      $config['conference_url'] = $conference_url;
    }

    $event_data = $this->googleCalendarService->updateEvent($auth_type, $client_id, $event_id, $config);

    if (!$event_data) {
      $this->logger->error('Google Calendar update event action: Failed to update event @event_id using @auth_type.', [
        '@event_id' => $event_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    if (!empty($event_token_name)) {
      $this->tokenService->addTokenData($event_token_name, $event_data);
    }
  }

}
