<?php

declare(strict_types=1);

namespace Drupal\eca_google_docs;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\eca_google\GoogleApiService;
use Google\Service\Docs;
use Google\Service\Drive;

/**
 * Service for Google Docs API operations.
 */
class GoogleDocsService {

  /**
   * The Google API service.
   */
  protected GoogleApiService $googleApiService;

  /**
   * The logger service.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a GoogleDocsService object.
   *
   * @param \Drupal\eca_google\GoogleApiService $google_api_service
   *   The Google API service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   */
  public function __construct(GoogleApiService $google_api_service, LoggerChannelFactoryInterface $logger_factory) {
    $this->googleApiService = $google_api_service;
    $this->logger = $logger_factory->get('eca_google_docs');
  }

  /**
   * Validates Google Drive API access.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   *
   * @return bool
   *   TRUE if Drive API access is available, FALSE otherwise.
   */
  public function validateDriveAccess(string $auth_type, string $client_id): bool {
    try {
      $drive_service = $this->googleApiService->getService('drive', $auth_type, $client_id);
      return $drive_service instanceof Drive;
    }
    catch (\Exception $e) {
      $this->logger->error('Drive API access validation failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Validates Google Docs API access.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   *
   * @return bool
   *   TRUE if Docs API access is available, FALSE otherwise.
   */
  public function validateDocsAccess(string $auth_type, string $client_id): bool {
    try {
      $docs_service = $this->googleApiService->getService('docs', $auth_type, $client_id);
      return $docs_service instanceof Docs;
    }
    catch (\Exception $e) {
      $this->logger->error('Docs API access validation failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Creates a new Google Docs document.
   *
   * @param string $auth_type
   *   The authentication type (oauth2 or service_account).
   * @param string $client_id
   *   The client ID for authentication.
   * @param array $config
   *   Configuration array with document properties.
   *
   * @return array|null
   *   Document data array or NULL on failure.
   */
  public function createDocument(string $auth_type, string $client_id, array $config): ?array {
    try {
      $service = $this->googleApiService->getService('docs', $auth_type, $client_id);
      if (!$service instanceof Docs) {
        $this->logger->error('Failed to get Google Docs service for document creation.');
        return NULL;
      }

      // Create document with title
      $document = new Docs\Document();
      $document->setTitle($config['title'] ?? 'Untitled Document');

      $created_document = $service->documents->create($document);

      if (!$created_document) {
        $this->logger->error('Failed to create Google Docs document.');
        return NULL;
      }

      // If initial content is provided, add it
      if (!empty($config['content'])) {
        $this->insertText($auth_type, $client_id, $created_document->getDocumentId(), $config['content'], 'position', 1);
      }

      return $this->formatDocumentResult($created_document, TRUE);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create Google Docs document: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Creates a document from a template with placeholder replacement.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID for authentication.
   * @param string $template_id
   *   The template document ID.
   * @param string $title
   *   The new document title.
   * @param array $placeholders
   *   Key-value pairs for placeholder replacement.
   *
   * @return array|null
   *   Document data array or NULL on failure.
   */
  public function createFromTemplate(string $auth_type, string $client_id, string $template_id, string $title, array $placeholders = []): ?array {
    try {
      // Validate Drive API access first
      if (!$this->validateDriveAccess($auth_type, $client_id)) {
        $this->logger->error('Google Drive API access is required for template document creation but is not available. Please ensure the Google client has Drive API enabled and proper scopes.');
        return NULL;
      }

      $drive_service = $this->googleApiService->getService('drive', $auth_type, $client_id);
      if (!$drive_service instanceof Drive) {
        $this->logger->error('Failed to get Google Drive service for template document creation.');
        return NULL;
      }
      $copied_file = new Drive\DriveFile();
      $copied_file->setName($title);

      $new_file = $drive_service->files->copy($template_id, $copied_file);

      if (!$new_file) {
        $this->logger->error('Failed to copy template document.');
        return NULL;
      }

      $document_id = $new_file->getId();

      // Replace placeholders if provided
      if (!empty($placeholders)) {
        $replacement_count = 0;
        foreach ($placeholders as $placeholder => $value) {
          // Handle HTML formatting and line breaks in placeholder values
          $formatted_value = $this->prepareReplacementText((string) $value);
          $result = $this->replaceText($auth_type, $client_id, $document_id, $placeholder, $formatted_value);
          if ($result && isset($result['replacement_count'])) {
            $replacement_count += $result['replacement_count'];
          }
        }
      }

      // Get the final document
      $docs_service = $this->googleApiService->getService('docs', $auth_type, $client_id);
      if (!$docs_service instanceof Docs) {
        $this->logger->error('Failed to get Google Docs service for retrieving final document.');
        return NULL;
      }
      $document = $docs_service->documents->get($document_id);

      $result = $this->formatDocumentResult($document, TRUE);
      $result['replacement_count'] = $replacement_count ?? 0;
      return $result;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create document from template: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Retrieves a Google Docs document.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID for authentication.
   * @param string $document_id
   *   The document ID to retrieve.
   *
   * @return array|null
   *   Document data array or NULL on failure.
   */
  public function getDocument(string $auth_type, string $client_id, string $document_id): ?array {
    try {
      $service = $this->googleApiService->getService('docs', $auth_type, $client_id);
      if (!$service instanceof Docs) {
        $this->logger->error('Failed to get Google Docs service for document retrieval.');
        return NULL;
      }
      $document = $service->documents->get($document_id);

      if (!$document) {
        $this->logger->error('Failed to retrieve document: @document_id', [
          '@document_id' => $document_id,
        ]);
        return NULL;
      }

      return $this->formatDocumentResult($document, FALSE);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve Google Docs document: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Inserts text into a document at a specific position or mode.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID for authentication.
   * @param string $document_id
   *   The document ID.
   * @param string $text
   *   The text to insert (can include HTML markup).
   * @param string $insert_mode
   *   The insertion mode: 'beginning', 'end', or 'position'.
   * @param int $position
   *   The position to insert at (1-based). Only used for 'position' mode.
   *
   * @return array|null
   *   Document data with insertion success status, or null on failure.
   */
  public function insertText(string $auth_type, string $client_id, string $document_id, string $text, string $insert_mode = 'end', int $position = 1): ?array {
    try {
      $service = $this->googleApiService->getService('docs', $auth_type, $client_id);
      if (!$service instanceof Docs) {
        $this->logger->error('Failed to get Google Docs service for text insertion.');
        return NULL;
      }

      // Get document first to determine insertion position
      if ($insert_mode === 'beginning') {
        $insert_position = 1;
      } elseif ($insert_mode === 'end') {
        $document = $service->documents->get($document_id);
        if (!$document || !$document->getBody()) {
          $this->logger->error('Failed to get document for end insertion.');
          return NULL;
        }
        $content = $document->getBody()->getContent();
        $insert_position = end($content)->getEndIndex() - 1; // -1 to insert before final newline
      } else {
        $insert_position = $position;
      }

      // Convert HTML to Google Docs requests
      $requests = $this->convertHtmlToDocsRequests($text, $insert_position);

      if (empty($requests)) {
        // Fallback to plain text if no HTML
        $requests = [
          new Docs\Request([
            'insertText' => [
              'location' => ['index' => $insert_position],
              'text' => $text,
            ],
          ]),
        ];
      }

      $batch_update_request = new Docs\BatchUpdateDocumentRequest([
        'requests' => $requests,
      ]);

      $response = $service->documents->batchUpdate($document_id, $batch_update_request);

      if ($response === NULL) {
        return NULL;
      }

      // Get updated document and return formatted result
      $updated_document = $service->documents->get($document_id);
      if (!$updated_document) {
        return NULL;
      }

      $result = $this->formatDocumentResult($updated_document);
      $result['insertion_success'] = TRUE;

      return $result;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to insert text into document: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Replaces text in a document.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID for authentication.
   * @param string $document_id
   *   The document ID.
   * @param string $search_text
   *   The text to search for.
   * @param string $replacement_text
   *   The replacement text (can include HTML markup).
   * @param bool $match_case
   *   Whether to match case.
   * @param bool $match_whole_word
   *   Whether to match whole words only.
   * @param bool $use_regex
   *   Whether to treat search text as a regular expression.
   *
   * @return array|null
   *   Document data with replacement count, or null on failure.
   */
  public function replaceText(string $auth_type, string $client_id, string $document_id, string $search_text, string $replacement_text, bool $match_case = FALSE, bool $match_whole_word = FALSE, bool $use_regex = FALSE): ?array {
    try {
      $service = $this->googleApiService->getService('docs', $auth_type, $client_id);
      if (!$service instanceof Docs) {
        $this->logger->error('Failed to get Google Docs service for text replacement.');
        return NULL;
      }

      // Configure search text and options
      $search_pattern = $search_text;
      $enable_regex = $use_regex;

      if ($match_whole_word && !$use_regex) {
        // Use regex word boundaries for whole word matching (only if not already using regex)
        $search_pattern = '\\b' . preg_quote($search_text, '/') . '\\b';
        $enable_regex = TRUE;
      }

      $contains_text_config = [
        'text' => $search_pattern,
        'matchCase' => $match_case,
        'searchByRegex' => $enable_regex,
      ];

      // For HTML replacement text, convert to plain text preserving line breaks
      if (strip_tags($replacement_text) !== $replacement_text) {
        // HTML content detected - convert to plain text but preserve line breaks
        $plain_replacement = $this->convertHtmlToPlainText($replacement_text);
        $requests = [
          new Docs\Request([
            'replaceAllText' => [
              'containsText' => $contains_text_config,
              'replaceText' => $plain_replacement,
            ],
          ]),
        ];
      } else {
        // Simple text replacement
        $requests = [
          new Docs\Request([
            'replaceAllText' => [
              'containsText' => $contains_text_config,
              'replaceText' => $replacement_text,
            ],
          ]),
        ];
      }

      $batch_update_request = new Docs\BatchUpdateDocumentRequest([
        'requests' => $requests,
      ]);

      $response = $service->documents->batchUpdate($document_id, $batch_update_request);

      if (!$response) {
        return NULL;
      }

      // Extract replacement count from response
      $replacement_count = 0;
      if ($response->getReplies()) {
        $reply = $response->getReplies()[0];
        if ($reply && $reply->getReplaceAllText()) {
          $replacement_count = $reply->getReplaceAllText()->getOccurrencesChanged() ?? 0;
        }
      }

      // Get updated document and return formatted result
      $updated_document = $service->documents->get($document_id);
      if (!$updated_document) {
        return NULL;
      }

      $result = $this->formatDocumentResult($updated_document);
      $result['replacement_count'] = $replacement_count;

      return $result;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to replace text in document: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Formats document result data into standardized array.
   *
   * @param Docs\Document $document
   *   The document object.
   * @param bool $include_created
   *   Whether to include creation timestamp.
   *
   * @return array
   *   Formatted document data.
   */
  protected function formatDocumentResult(Docs\Document $document, bool $include_created = FALSE): array {
    $formatted = [
      'document_id' => $document->getDocumentId(),
      'title' => $document->getTitle(),
      'revision_id' => $document->getRevisionId(),
      'document_url' => 'https://docs.google.com/document/d/' . $document->getDocumentId(),
    ];

    // Extract both plain text and HTML content
    $plain_content = '';
    $html_content = '';

    if ($document->getBody() && $document->getBody()->getContent()) {
      foreach ($document->getBody()->getContent() as $element) {
        if ($element->getParagraph()) {
          $paragraph_text = '';
          $paragraph_html = '<p>';

          foreach ($element->getParagraph()->getElements() as $paragraph_element) {
            if ($paragraph_element->getTextRun()) {
              $text = $paragraph_element->getTextRun()->getContent();
              $plain_content .= $text;
              $paragraph_text .= $text;

              // Build HTML with formatting
              $text_style = $paragraph_element->getTextRun()->getTextStyle();
              $formatted_text = $this->applyHtmlFormatting($text, $text_style);
              $paragraph_html .= $formatted_text;
            }
          }

          $paragraph_html .= '</p>';

          // Only add paragraph HTML if it has content beyond <p></p>
          if ($paragraph_text) {
            $html_content .= $paragraph_html;
          }
        }
      }
    }

    $formatted['content'] = trim($plain_content);
    $formatted['content_html'] = trim($html_content);
    $formatted['word_count'] = str_word_count($formatted['content']);

    if ($include_created) {
      $formatted['created'] = date('c');
    }

    return $formatted;
  }

  /**
   * Converts HTML markup to Google Docs API requests.
   *
   * @param string $html_text
   *   Text that may contain HTML markup.
   * @param int $start_position
   *   The starting position for insertion.
   *
   * @return array
   *   Array of Google Docs API request objects.
   */
  protected function convertHtmlToDocsRequests(string $html_text, int $start_position): array {
    // Check if text contains HTML tags
    if (strip_tags($html_text) === $html_text) {
      return []; // No HTML, return empty to use fallback
    }

    $requests = [];
    $current_position = $start_position;

    // Convert HTML to plain text but preserve formatting info
    $dom = new \DOMDocument();
    $dom->loadHTML('<?xml encoding="UTF-8">' . $html_text, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
    $xpath = new \DOMXPath($dom);

    // Extract text content and build formatting requests
    $text_content = $this->extractTextWithFormatting($dom->documentElement, $current_position, $requests);

    if (!empty($text_content)) {
      // Insert plain text first
      array_unshift($requests, new Docs\Request([
        'insertText' => [
          'location' => ['index' => $start_position],
          'text' => $text_content,
        ],
      ]));
    }

    return $requests;
  }

  /**
   * Recursively extracts text and builds formatting requests.
   *
   * @param \DOMNode $node
   *   The DOM node to process.
   * @param int &$current_position
   *   Current position in the document (passed by reference).
   * @param array &$requests
   *   Array to collect formatting requests (passed by reference).
   *
   * @return string
   *   Extracted plain text.
   */
  protected function extractTextWithFormatting(\DOMNode $node, int &$current_position, array &$requests): string {
    $text = '';

    foreach ($node->childNodes as $child) {
      if ($child->nodeType === XML_TEXT_NODE) {
        $node_text = $child->textContent;
        $text .= $node_text;

        // Apply formatting based on parent elements
        $text_style = $this->getTextStyleFromNode($child->parentNode);
        if (!empty($text_style)) {
          $requests[] = new Docs\Request([
            'updateTextStyle' => [
              'range' => [
                'startIndex' => $current_position,
                'endIndex' => $current_position + mb_strlen($node_text),
              ],
              'textStyle' => $text_style,
              'fields' => implode(',', array_keys($text_style)),
            ],
          ]);
        }

        $current_position += mb_strlen($node_text);
      }
      elseif ($child->nodeType === XML_ELEMENT_NODE) {
        if (strtolower($child->nodeName) === 'p') {
          // Add paragraph break
          $paragraph_text = $this->extractTextWithFormatting($child, $current_position, $requests);
          $text .= $paragraph_text . "\n";
          $current_position += 1; // For the newline
        }
        else {
          $text .= $this->extractTextWithFormatting($child, $current_position, $requests);
        }
      }
    }

    return $text;
  }

  /**
   * Determines text style from DOM node.
   *
   * @param \DOMNode $node
   *   The DOM node to analyze.
   *
   * @return array
   *   Google Docs text style properties.
   */
  protected function getTextStyleFromNode(\DOMNode $node): array {
    $style = [];

    // Walk up the DOM tree to collect all formatting
    $current = $node;
    while ($current && $current->nodeType === XML_ELEMENT_NODE) {
      switch (strtolower($current->nodeName)) {
        case 'strong':
        case 'b':
          $style['bold'] = TRUE;
          break;
        case 'em':
        case 'i':
          $style['italic'] = TRUE;
          break;
        case 'u':
          $style['underline'] = TRUE;
          break;
      }
      $current = $current->parentNode;
    }

    return $style;
  }

  /**
   * Applies HTML formatting based on Google Docs text style.
   *
   * @param string $text
   *   The text content.
   * @param Docs\TextStyle|null $text_style
   *   The Google Docs text style object.
   *
   * @return string
   *   Text wrapped in appropriate HTML tags.
   */
  protected function applyHtmlFormatting(string $text, $text_style): string {
    if (!$text_style) {
      return htmlspecialchars($text);
    }

    $formatted_text = htmlspecialchars($text);

    // Apply formatting in reverse order so tags nest properly
    if ($text_style->getUnderline()) {
      $formatted_text = '<u>' . $formatted_text . '</u>';
    }

    if ($text_style->getItalic()) {
      $formatted_text = '<em>' . $formatted_text . '</em>';
    }

    if ($text_style->getBold()) {
      $formatted_text = '<strong>' . $formatted_text . '</strong>';
    }

    return $formatted_text;
  }

  /**
   * Prepares replacement text for placeholder substitution.
   *
   * Handles HTML formatting and line breaks in replacement values.
   *
   * @param string $text
   *   The replacement text that may contain HTML or line breaks.
   *
   * @return string
   *   Prepared text suitable for Google Docs insertion.
   */
  protected function prepareReplacementText(string $text): string {
    if (empty($text)) {
      return $text;
    }

    // First, decode HTML entities (handles &lt;p&gt; -> <p> conversion)
    $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');

    // Handle line breaks - convert various line break formats to \n
    $text = str_replace(["\r\n", "\r"], "\n", $text);
    
    // Check if text contains HTML tags (after entity decoding)
    if (strip_tags($text) !== $text) {
      // Text contains HTML - let replaceText handle the conversion
      return $text;
    }
    
    // For plain text with line breaks, convert \n to <br> for HTML processing
    if (strpos($text, "\n") !== FALSE) {
      $text = nl2br($text, false); // Convert to <br> tags without XHTML
    }
    
    return $text;
  }

  /**
   * Converts HTML content to plain text preserving line breaks.
   *
   * @param string $html
   *   HTML content that may contain formatting and line breaks.
   *
   * @return string
   *   Plain text with line breaks preserved.
   */
  protected function convertHtmlToPlainText(string $html): string {
    if (empty($html)) {
      return $html;
    }

    // Convert HTML line break elements to actual line breaks
    $html = str_ireplace(['<br>', '<br/>', '<br />'], "\n", $html);
    
    // Convert paragraph tags to line breaks (with double line breaks for paragraph separation)
    $html = preg_replace('/<\/p>\s*<p[^>]*>/i', "\n\n", $html);
    $html = preg_replace('/<\/?p[^>]*>/i', "\n", $html);
    
    // Convert div tags to line breaks
    $html = preg_replace('/<\/div>\s*<div[^>]*>/i', "\n", $html);
    $html = preg_replace('/<\/?div[^>]*>/i', "\n", $html);
    
    // Strip remaining HTML tags
    $text = strip_tags($html);
    
    // Clean up excessive line breaks
    $text = preg_replace('/\n{3,}/', "\n\n", $text);
    $text = trim($text);
    
    return $text;
  }

}
