<?php

declare(strict_types=1);

namespace Drupal\eca_google_docs\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Create a new Google Docs document.
 */
#[Action(
  id: 'eca_google_docs_create_document',
  label: new TranslatableMarkup('Google Docs: Create Document'),
  category: new TranslatableMarkup('Google Docs'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Creates a new Google Docs document with optional initial content and template support.'),
  version_introduced: '1.0.0',
)]
final class CreateDocument extends DocsActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'title' => '',
      'content' => '',
      'document_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Document Title'),
      '#description' => $this->t('The title for the new document.'),
      '#default_value' => $this->configuration['title'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['content'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Initial Content'),
      '#description' => $this->t('Optional initial text content for the document. Leave empty to create a blank document.'),
      '#default_value' => $this->configuration['content'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['document_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Document Token Name'),
      '#description' => $this->t('Optional token name to store document details. Accessible as [token_name:document_id], [token_name:document_url], [token_name:title], [token_name:content].'),
      '#default_value' => $this->configuration['document_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }


  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $title = $this->tokenService->replacePlain($this->configuration['title']);
    $content = $this->tokenService->replace($this->configuration['content'], [], ['clear' => TRUE]);
    $document_token_name = $this->tokenService->replacePlain($this->configuration['document_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('create document');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($title)) {
      $this->logger->error('Google Docs create document action: Missing document title.');
      return;
    }


    // Create new blank document
    $config = [
      'title' => $title,
      'content' => $content,
    ];

    $document_data = $this->googleDocsService->createDocument($auth_type, $client_id, $config);

    if (!$document_data) {
      $this->logger->error('Google Docs create document action: Failed to create document using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store document data in token if requested
    if (!empty($document_token_name)) {
      $this->tokenService->addTokenData($document_token_name, $document_data);
    }
  }

}
