<?php

declare(strict_types=1);

namespace Drupal\eca_google_docs\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Create a Google Docs document from a template.
 */
#[Action(
  id: 'eca_google_docs_create_from_template',
  label: new TranslatableMarkup('Google Docs: Create from Template'),
  category: new TranslatableMarkup('Google Docs'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Creates a Google Docs document by copying a template and replacing placeholder variables with dynamic data.'),
  version_introduced: '1.0.0',
)]
final class CreateFromTemplate extends DocsActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'template_id' => '',
      'title' => '',
      'placeholders' => '',
      'document_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['template_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Template Document ID'),
      '#description' => $this->t('The ID of the Google Docs template document to copy. <strong>Note:</strong> This requires Google Drive API access in addition to Google Docs API.'),
      '#default_value' => $this->configuration['template_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('New Document Title'),
      '#description' => $this->t('The title for the new document created from the template.'),
      '#default_value' => $this->configuration['title'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['placeholders'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Template Variables'),
      '#description' => $this->t('Key-value pairs for template placeholder replacement. One per line in format: <code>placeholder_name: replacement_value</code><br>Example:<br><code>client_name: [node:field_client_name]<br>project_date: [node:field_start_date]</code><br>These will replace <code>{{placeholder_name}}</code> in the template.'),
      '#default_value' => $this->configuration['placeholders'],
      '#rows' => 5,
      '#eca_token_replacement' => TRUE,
    ];

    $form['document_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Document Token Name'),
      '#description' => $this->t('Optional token name to store document details. Accessible as [token_name:document_id], [token_name:document_url], [token_name:title], [token_name:replacement_count].'),
      '#default_value' => $this->configuration['document_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);

    // Always validate Drive API access for template functionality
    $auth_client_id = $form_state->getValue('auth_client_id');
    if (!empty($auth_client_id)) {
      $auth_info = $this->googleApiService->parseAuthClientId($auth_client_id);
      if ($auth_info) {
        [$auth_type, $client_id] = [$auth_info['auth_type'], $auth_info['client_id']];

        if (!$this->googleDocsService->validateDriveAccess($auth_type, $client_id)) {
          $form_state->setErrorByName('auth_client_id', $this->t('Template document creation requires Google Drive API access. Please ensure the selected Google API client has Drive API enabled.'));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $template_id = $this->tokenService->replacePlain($this->configuration['template_id']);
    $title = $this->tokenService->replacePlain($this->configuration['title']);
    $placeholders_text = $this->configuration['placeholders'];
    $document_token_name = $this->tokenService->replacePlain($this->configuration['document_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('create from template');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (!$this->validateDocumentId($template_id, 'create from template')) {
      return;
    }

    if (empty($title)) {
      $this->logger->error('Google Docs create from template action: Missing document title.');
      return;
    }


    // Parse placeholders from text format
    $placeholders = $this->parsePlaceholders($placeholders_text);

    $document_data = $this->googleDocsService->createFromTemplate(
      $auth_type,
      $client_id,
      $template_id,
      $title,
      $placeholders
    );

    if (!$document_data) {
      $this->logger->error('Google Docs create from template action: Failed to create document from template @template_id using @auth_type.', [
        '@template_id' => $template_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store document data in token if requested
    if (!empty($document_token_name)) {
      $this->tokenService->addTokenData($document_token_name, $document_data);
    }
  }

  /**
   * Parses placeholder text into key-value array.
   *
   * @param string $placeholders_text
   *   Text with placeholder definitions, one per line.
   *
   * @return array
   *   Array of placeholder key-value pairs.
   */
  protected function parsePlaceholders(string $placeholders_text): array {
    $placeholders = [];

    if (empty($placeholders_text)) {
      return $placeholders;
    }

    $lines = preg_split('/\r\n|\r|\n/', $placeholders_text);
    foreach ($lines as $line) {
      $line = trim($line);
      if (empty($line) || strpos($line, ':') === FALSE) {
        continue;
      }

      [$key, $value] = explode(':', $line, 2);
      $key = trim($key);
      $value = trim($value);

      if (!empty($key)) {
        // Replace tokens in the value, preserving HTML formatting
        $replaced_value = $this->tokenService->replace($value, [], ['clear' => TRUE]);
        
        // Add braces to make it a proper template placeholder
        $placeholder_key = '{{' . $key . '}}';
        $placeholders[$placeholder_key] = $replaced_value;
      }
    }

    return $placeholders;
  }

}
