<?php

declare(strict_types=1);

namespace Drupal\eca_google_docs\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca_google\Plugin\Action\GoogleApiActionBase;
use Drupal\eca_google_docs\GoogleDocsService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Google Docs ECA actions.
 */
abstract class DocsActionBase extends GoogleApiActionBase {

  /**
   * The Google Docs service.
   */
  protected GoogleDocsService $googleDocsService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->googleDocsService = $container->get('eca_google_docs.google_docs');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);
    $this->validateDocsAccess($form, $form_state);
  }

  /**
   * Validates Google Docs API access.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  protected function validateDocsAccess(array $form, FormStateInterface $form_state): void {
    $auth_client_id = $form_state->getValue('auth_client_id');
    if (!empty($auth_client_id)) {
      $auth_info = $this->googleApiService->parseAuthClientId($auth_client_id);
      if ($auth_info) {
        [$auth_type, $client_id] = [$auth_info['auth_type'], $auth_info['client_id']];

        if (!$this->googleDocsService->validateDocsAccess($auth_type, $client_id)) {
          $form_state->setErrorByName('auth_client_id', $this->t('The selected Google API client does not have access to Google Docs API.'));
        }
      }
    }
  }

  /**
   * Validates and parses authentication client ID.
   *
   * @param string $auth_client_id
   *   The auth client ID to validate and parse.
   * @param string $action_name
   *   The action name for error messages.
   *
   * @return array|null
   *   Array with 'auth_type' and 'client_id' keys, or NULL on failure.
   */

  /**
   * Validates document ID format.
   *
   * @param string $document_id
   *   The document ID to validate.
   * @param string $operation
   *   The operation being performed (for error messages).
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  protected function validateDocumentId(string $document_id, string $operation): bool {
    if (empty($document_id)) {
      $this->logger->error('Google Docs @operation action: Missing document_id.', [
        '@operation' => $operation,
      ]);
      return FALSE;
    }

    // Basic document ID validation - Google Doc IDs are typically 44 characters
    if (strlen($document_id) < 20 || !preg_match('/^[a-zA-Z0-9_-]+$/', $document_id)) {
      $this->logger->error('Google Docs @operation action: Invalid document_id format: @document_id', [
        '@operation' => $operation,
        '@document_id' => $document_id,
      ]);
      return FALSE;
    }

    return TRUE;
  }

}