<?php

declare(strict_types=1);

namespace Drupal\eca_google_docs\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Retrieve a Google Docs document.
 */
#[Action(
  id: 'eca_google_docs_get_document',
  label: new TranslatableMarkup('Google Docs: Get Document'),
  category: new TranslatableMarkup('Google Docs'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieves a Google Docs document content and metadata for processing or analysis.'),
  version_introduced: '1.0.0',
)]
final class GetDocument extends DocsActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'document_id' => '',
      'document_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['document_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Document ID'),
      '#description' => $this->t('The ID of the Google Docs document to retrieve.'),
      '#default_value' => $this->configuration['document_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['document_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Document Token Name'),
      '#description' => $this->t('Optional token name to store document details. Accessible as [token_name:document_id], [token_name:title], [token_name:content], [token_name:content_html], [token_name:word_count], [token_name:revision_id].'),
      '#default_value' => $this->configuration['document_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $document_id = $this->tokenService->replacePlain($this->configuration['document_id']);
    $document_token_name = $this->tokenService->replacePlain($this->configuration['document_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('get document');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (!$this->validateDocumentId($document_id, 'get document')) {
      return;
    }


    $document_data = $this->googleDocsService->getDocument($auth_type, $client_id, $document_id);

    if (!$document_data) {
      $this->logger->error('Google Docs get document action: Failed to retrieve document @document_id using @auth_type.', [
        '@document_id' => $document_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store document data in token if requested
    if (!empty($document_token_name)) {
      $this->tokenService->addTokenData($document_token_name, $document_data);
    }
  }

}
