<?php

declare(strict_types=1);

namespace Drupal\eca_google_docs\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Insert text content into a Google Docs document.
 */
#[Action(
  id: 'eca_google_docs_insert_text',
  label: new TranslatableMarkup('Google Docs: Insert Text'),
  category: new TranslatableMarkup('Google Docs'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Inserts text content at a specific position in a Google Docs document with optional formatting.'),
  version_introduced: '1.0.0',
)]
final class InsertText extends DocsActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'document_id' => '',
      'text_content' => '',
      'position' => '1',
      'insert_mode' => 'end',
      'document_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['document_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Document ID'),
      '#description' => $this->t('The ID of the Google Docs document to modify.'),
      '#default_value' => $this->configuration['document_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['text_content'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Text Content'),
      '#description' => $this->t('The text content to insert into the document. Supports basic HTML formatting (bold, italic, underline).'),
      '#default_value' => $this->configuration['text_content'],
      '#required' => TRUE,
      '#rows' => 3,
      '#eca_token_replacement' => TRUE,
    ];

    $form['insert_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Insert Mode'),
      '#description' => $this->t('Where to insert the text in the document.'),
      '#default_value' => $this->configuration['insert_mode'],
      '#options' => [
        'beginning' => $this->t('At beginning of document'),
        'end' => $this->t('At end of document'),
        'position' => $this->t('At specific position'),
      ],
      '#required' => TRUE,
    ];

    $form['position'] = [
      '#type' => 'number',
      '#title' => $this->t('Position'),
      '#description' => $this->t('Character position for insertion (1-based). Only used when Insert Mode is "At specific position".'),
      '#default_value' => $this->configuration['position'],
      '#min' => 1,
      '#states' => [
        'visible' => [
          ':input[name="insert_mode"]' => ['value' => 'position'],
        ],
        'required' => [
          ':input[name="insert_mode"]' => ['value' => 'position'],
        ],
      ],
      '#eca_token_replacement' => TRUE,
    ];

    $form['document_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Document Token Name'),
      '#description' => $this->t('Optional token name to store updated document details. Accessible as [token_name:document_id], [token_name:title], [token_name:content], [token_name:word_count], [token_name:insertion_success].'),
      '#default_value' => $this->configuration['document_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $document_id = $this->tokenService->replacePlain($this->configuration['document_id']);
    $text_content = $this->tokenService->replace($this->configuration['text_content']);
    $insert_mode = $this->configuration['insert_mode'];
    $position = (int) $this->tokenService->replacePlain((string) $this->configuration['position']);
    $document_token_name = $this->tokenService->replacePlain($this->configuration['document_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('insert text');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (!$this->validateDocumentId($document_id, 'insert text')) {
      return;
    }

    if (empty($text_content)) {
      $this->logger->error('Google Docs insert text action: Missing text content to insert.');
      return;
    }


    $document_data = $this->googleDocsService->insertText(
      $auth_type,
      $client_id,
      $document_id,
      $text_content,
      $insert_mode,
      $position
    );

    if (!$document_data) {
      $this->logger->error('Google Docs insert text action: Failed to insert text into document @document_id using @auth_type.', [
        '@document_id' => $document_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store document data in token if requested
    if (!empty($document_token_name)) {
      $this->tokenService->addTokenData($document_token_name, $document_data);
    }
  }

}