<?php

declare(strict_types=1);

namespace Drupal\eca_google_docs\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Find and replace text patterns in a Google Docs document.
 */
#[Action(
  id: 'eca_google_docs_replace_text',
  label: new TranslatableMarkup('Google Docs: Replace Text'),
  category: new TranslatableMarkup('Google Docs'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Finds and replaces text patterns throughout a Google Docs document with match options.'),
  version_introduced: '1.0.0',
)]
final class ReplaceText extends DocsActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'document_id' => '',
      'search_text' => '',
      'replacement_text' => '',
      'match_case' => FALSE,
      'match_whole_word' => FALSE,
      'use_regex' => FALSE,
      'document_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['document_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Document ID'),
      '#description' => $this->t('The ID of the Google Docs document to modify.'),
      '#default_value' => $this->configuration['document_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['search_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search Text'),
      '#description' => $this->t('The text or pattern to search for in the document.'),
      '#default_value' => $this->configuration['search_text'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['replacement_text'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Replacement Text'),
      '#description' => $this->t('The text to replace matches with. Supports basic HTML formatting (bold, italic, underline). Leave empty to delete matches.'),
      '#default_value' => $this->configuration['replacement_text'],
      '#rows' => 3,
      '#eca_token_replacement' => TRUE,
    ];

    $form['match_case'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Match case'),
      '#description' => $this->t('Perform case-sensitive matching.'),
      '#default_value' => $this->configuration['match_case'],
    ];

    $form['match_whole_word'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Match whole words only'),
      '#description' => $this->t('Only match complete words, not partial matches.'),
      '#default_value' => $this->configuration['match_whole_word'],
    ];

    $form['use_regex'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Treat search text as regular expression'),
      '#description' => $this->t('Enable regex pattern matching. When enabled, search text will be treated as a regular expression pattern.'),
      '#default_value' => $this->configuration['use_regex'],
    ];

    $form['document_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Document Token Name'),
      '#description' => $this->t('Optional token name to store updated document details. Accessible as [token_name:document_id], [token_name:title], [token_name:content], [token_name:word_count], [token_name:replacement_count].'),
      '#default_value' => $this->configuration['document_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $document_id = $this->tokenService->replacePlain($this->configuration['document_id']);
    $search_text = $this->tokenService->replacePlain($this->configuration['search_text']);
    $replacement_text = $this->tokenService->replace($this->configuration['replacement_text']);
    $match_case = (bool) $this->configuration['match_case'];
    $match_whole_word = (bool) $this->configuration['match_whole_word'];
    $use_regex = (bool) $this->configuration['use_regex'];
    $document_token_name = $this->tokenService->replacePlain($this->configuration['document_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('replace text');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (!$this->validateDocumentId($document_id, 'replace text')) {
      return;
    }

    if (empty($search_text)) {
      $this->logger->error('Google Docs replace text action: Missing search text.');
      return;
    }


    $document_data = $this->googleDocsService->replaceText(
      $auth_type,
      $client_id,
      $document_id,
      $search_text,
      $replacement_text,
      $match_case,
      $match_whole_word,
      $use_regex
    );

    if (!$document_data) {
      $this->logger->error('Google Docs replace text action: Failed to replace text in document @document_id using @auth_type.', [
        '@document_id' => $document_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store document data in token if requested
    if (!empty($document_token_name)) {
      $this->tokenService->addTokenData($document_token_name, $document_data);
    }
  }

}
