<?php

declare(strict_types=1);

namespace Drupal\eca_google_drive;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\eca_google\GoogleApiService;
use Google\Service\Drive;
use Google\Service\Drive\DriveFile;
use Google\Service\Drive\Permission;

/**
 * Service for Google Drive API operations.
 */
class GoogleDriveService {

  use StringTranslationTrait;

  // Permission role constants
  public const string ROLE_READER = 'reader';
  public const string ROLE_COMMENTER = 'commenter';
  public const string ROLE_WRITER = 'writer';
  public const string ROLE_OWNER = 'owner';

  // Permission type constants
  public const string TYPE_USER = 'user';
  public const string TYPE_GROUP = 'group';
  public const string TYPE_DOMAIN = 'domain';
  public const string TYPE_ANYONE = 'anyone';

  // Google Workspace MIME types
  public const string MIME_GOOGLE_DOCS = 'application/vnd.google-apps.document';
  public const string MIME_GOOGLE_SHEETS = 'application/vnd.google-apps.spreadsheet';
  public const string MIME_GOOGLE_SLIDES = 'application/vnd.google-apps.presentation';
  public const string MIME_FOLDER = 'application/vnd.google-apps.folder';

  // Export format constants
  public const string EXPORT_PDF = 'application/pdf';
  public const string EXPORT_DOCX = 'application/vnd.openxmlformats-officedocument.wordprocessingml.document';
  public const string EXPORT_XLSX = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
  public const string EXPORT_PPTX = 'application/vnd.openxmlformats-officedocument.presentationml.presentation';
  public const string EXPORT_CSV = 'text/csv';
  public const string EXPORT_HTML = 'text/html';
  public const string EXPORT_TXT = 'text/plain';

  // File size limits (in MB)
  public const int DEFAULT_SIZE_LIMIT = 50;

  /**
   * The Google API service.
   */
  protected GoogleApiService $googleApiService;

  /**
   * The logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a new GoogleDriveService.
   */
  public function __construct(GoogleApiService $google_api_service, LoggerChannelFactoryInterface $logger_factory) {
    $this->googleApiService = $google_api_service;
    $this->logger = $logger_factory->get('eca_google_drive');
  }

  /**
   * Gets available permission role options.
   */
  public function getPermissionRoleOptions(): array {
    return [
      self::ROLE_READER => $this->t('Reader (view only)'),
      self::ROLE_COMMENTER => $this->t('Commenter (view and comment)'),
      self::ROLE_WRITER => $this->t('Writer (edit, comment, and share)'),
      self::ROLE_OWNER => $this->t('Owner (full control)'),
    ];
  }

  /**
   * Gets available permission type options.
   */
  public function getPermissionTypeOptions(): array {
    return [
      self::TYPE_USER => $this->t('User (specific email)'),
      self::TYPE_GROUP => $this->t('Group (Google Group email)'),
      self::TYPE_DOMAIN => $this->t('Domain (entire domain)'),
      self::TYPE_ANYONE => $this->t('Anyone (public access)'),
    ];
  }

  /**
   * Gets available export format options for Google Workspace documents.
   */
  public function getExportFormatOptions(): array {
    return [
      '' => $this->t('- No conversion (original format) -'),
      self::EXPORT_PDF => $this->t('PDF (all document types)'),
      self::EXPORT_DOCX => $this->t('Microsoft Word (.docx) - Docs only'),
      self::EXPORT_XLSX => $this->t('Microsoft Excel (.xlsx) - Sheets only'),
      self::EXPORT_PPTX => $this->t('Microsoft PowerPoint (.pptx) - Slides only'),
      self::EXPORT_CSV => $this->t('CSV (.csv) - Sheets only'),
      self::EXPORT_HTML => $this->t('HTML (.html) - Docs and Sheets'),
      self::EXPORT_TXT => $this->t('Plain Text (.txt) - Docs and Slides'),
    ];
  }

  /**
   * Uploads a file to Google Drive.
   */
  public function uploadFile(string $auth_type, string $client_id, array $upload_params): ?array {
    try {
      $service = $this->googleApiService->getService('drive', $auth_type, $client_id);
      if (!$service instanceof Drive) {
        $this->logger->error('Failed to get Google Drive service for file upload.');
        return NULL;
      }

      // Validate file size if limit is specified
      $file_size = $upload_params['file_size'] ?? 0;
      $size_limit = $upload_params['size_limit'] ?? 0;

      if ($size_limit > 0 && $file_size > $size_limit) {
        $this->logger->error('File size (@size MB) exceeds limit (@limit MB)', [
          '@size' => round($file_size / 1048576, 2),
          '@limit' => round($size_limit / 1048576, 2),
        ]);
        return NULL;
      }

      // Create file metadata
      $file_metadata = new DriveFile();
      $file_metadata->setName($upload_params['name']);

      if (!empty($upload_params['parent_folder_id'])) {
        $file_metadata->setParents([$upload_params['parent_folder_id']]);
      }

      // Handle different upload sources
      $file_content = NULL;
      $mime_type = $upload_params['mime_type'] ?? 'application/octet-stream';

      if (!empty($upload_params['file_path'])) {
        // Upload from local file path
        $file_content = file_get_contents($upload_params['file_path']);
      }
      elseif (!empty($upload_params['file_content'])) {
        // Upload from content string
        $file_content = $upload_params['file_content'];
      }
      elseif (!empty($upload_params['url'])) {
        // Upload from URL
        $file_content = file_get_contents($upload_params['url']);
      }

      if ($file_content === NULL || $file_content === FALSE) {
        $this->logger->error('Failed to get file content for upload');
        return NULL;
      }

      // Upload file
      $uploaded_file = $service->files->create($file_metadata, [
        'data' => $file_content,
        'mimeType' => $mime_type,
        'uploadType' => 'multipart',
      ]);

      return $this->formatFileResult($uploaded_file);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to upload file to Google Drive: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Downloads a file from Google Drive.
   */
  public function downloadFile(string $auth_type, string $client_id, string $file_id, array $download_params = []): ?array {
    try {
      $service = $this->googleApiService->getService('drive', $auth_type, $client_id);
      if (!$service instanceof Drive) {
        $this->logger->error('Failed to get Google Drive service for file download.');
        return NULL;
      }

      // Get file metadata first
      $file = $service->files->get($file_id, ['fields' => 'id,name,mimeType,size']);

      $requested_export_format = $download_params['export_format'] ?? '';
      $file_content = NULL;
      $actual_export_format = NULL;

      // Handle Google Workspace documents with export
      if ($this->isGoogleWorkspaceDocument($file->getMimeType()) && !empty($requested_export_format)) {
        $response = $service->files->export($file_id, $requested_export_format);
        $stream = $response->getBody();
        $stream->rewind(); // Ensure we're at the beginning
        $file_content = $stream->read($stream->getSize());
        $actual_export_format = $requested_export_format; // Export actually happened
      }
      else {
        // Regular file download - no export format conversion
        $response = $service->files->get($file_id, ['alt' => 'media']);
        $stream = $response->getBody();
        $stream->rewind(); // Ensure we're at the beginning
        $file_content = $stream->read($stream->getSize());
        // $actual_export_format stays NULL - no export happened
      }

      if ($file_content === NULL) {
        $this->logger->error('Failed to get file content from Google Drive');
        return NULL;
      }

      return [
        'file_id' => $file->getId(),
        'name' => $file->getName(),
        'mime_type' => $file->getMimeType(),
        'size' => $file->getSize(),
        'content' => $file_content,
        'export_format' => $actual_export_format,
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to download file from Google Drive: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets file information from Google Drive.
   */
  public function getFileInfo(string $auth_type, string $client_id, string $file_id): ?array {
    try {
      $service = $this->googleApiService->getService('drive', $auth_type, $client_id);
      if (!$service instanceof Drive) {
        $this->logger->error('Failed to get Google Drive service for file info.');
        return NULL;
      }

      $file = $service->files->get($file_id, [
        'fields' => 'id,name,mimeType,size,createdTime,modifiedTime,owners,parents,webViewLink,webContentLink,shared,capabilities'
      ]);

      return $this->formatFileResult($file);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get file info from Google Drive: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Searches for files in Google Drive.
   */
  public function searchFiles(string $auth_type, string $client_id, array $search_params): ?array {
    try {
      $service = $this->googleApiService->getService('drive', $auth_type, $client_id);
      if (!$service instanceof Drive) {
        $this->logger->error('Failed to get Google Drive service for file search.');
        return NULL;
      }

      $params = [
        'fields' => 'nextPageToken,files(id,name,mimeType,size,createdTime,modifiedTime,owners,parents,webViewLink,shared)',
      ];

      // Build search query
      if (!empty($search_params['query'])) {
        $params['q'] = $search_params['query'];
      }

      if (!empty($search_params['max_results'])) {
        $params['pageSize'] = min((int) $search_params['max_results'], 1000);
      }

      if (!empty($search_params['page_token'])) {
        $params['pageToken'] = $search_params['page_token'];
      }

      $response = $service->files->listFiles($params);
      $files = [];

      foreach ($response->getFiles() as $file) {
        $files[] = $this->formatFileResult($file);
      }

      return [
        'files' => $files,
        'total_files' => count($files),
        'next_page_token' => $response->getNextPageToken(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to search files in Google Drive: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Creates a folder in Google Drive.
   */
  public function createFolder(string $auth_type, string $client_id, string $folder_name, ?string $parent_folder_id = NULL): ?array {
    try {
      $service = $this->googleApiService->getService('drive', $auth_type, $client_id);
      if (!$service instanceof Drive) {
        $this->logger->error('Failed to get Google Drive service for folder creation.');
        return NULL;
      }

      $folder_metadata = new DriveFile();
      $folder_metadata->setName($folder_name);
      $folder_metadata->setMimeType(self::MIME_FOLDER);

      if ($parent_folder_id) {
        $folder_metadata->setParents([$parent_folder_id]);
      }

      $folder = $service->files->create($folder_metadata);

      return $this->formatFileResult($folder);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create folder in Google Drive: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Shares a file or folder in Google Drive.
   */
  public function shareFile(string $auth_type, string $client_id, string $file_id, array $permission_params): ?array {
    try {
      $service = $this->googleApiService->getService('drive', $auth_type, $client_id);
      if (!$service instanceof Drive) {
        $this->logger->error('Failed to get Google Drive service for file sharing.');
        return NULL;
      }

      $permission = new Permission();
      $permission->setRole($permission_params['role']);
      $permission->setType($permission_params['type']);

      if (!empty($permission_params['email_address']) && in_array($permission_params['type'], [self::TYPE_USER, self::TYPE_GROUP])) {
        $permission->setEmailAddress($permission_params['email_address']);
      }

      if (!empty($permission_params['domain']) && $permission_params['type'] === self::TYPE_DOMAIN) {
        $permission->setDomain($permission_params['domain']);
      }

      $options = [];
      if (isset($permission_params['send_notification'])) {
        $options['sendNotificationEmail'] = (bool) $permission_params['send_notification'];
      }

      if (!empty($permission_params['message'])) {
        $options['emailMessage'] = $permission_params['message'];
      }

      $created_permission = $service->permissions->create($file_id, $permission, $options);

      return [
        'permission_id' => $created_permission->getId(),
        'role' => $created_permission->getRole(),
        'type' => $created_permission->getType(),
        'email_address' => $created_permission->getEmailAddress(),
        'domain' => $created_permission->getDomain(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to share file in Google Drive: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Copies a file in Google Drive.
   */
  public function copyFile(string $auth_type, string $client_id, string $file_id, array $copy_params): ?array {
    try {
      $service = $this->googleApiService->getService('drive', $auth_type, $client_id);
      if (!$service instanceof Drive) {
        $this->logger->error('Failed to get Google Drive service for file copy.');
        return NULL;
      }

      $copied_file_metadata = new DriveFile();
      $copied_file_metadata->setName($copy_params['name']);

      if (!empty($copy_params['parent_folder_id'])) {
        $copied_file_metadata->setParents([$copy_params['parent_folder_id']]);
      }

      $copied_file = $service->files->copy($file_id, $copied_file_metadata);

      return $this->formatFileResult($copied_file);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to copy file in Google Drive: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Moves a file in Google Drive.
   */
  public function moveFile(string $auth_type, string $client_id, string $file_id, array $move_params): ?array {
    try {
      $service = $this->googleApiService->getService('drive', $auth_type, $client_id);
      if (!$service instanceof Drive) {
        $this->logger->error('Failed to get Google Drive service for file move.');
        return NULL;
      }

      // Get current file to find current parents
      $file = $service->files->get($file_id, ['fields' => 'parents']);
      $previous_parents = implode(',', $file->getParents());

      $updated_file = new DriveFile();

      $params = [];
      if (!empty($move_params['parent_folder_id'])) {
        $params['addParents'] = $move_params['parent_folder_id'];
      }
      if ($previous_parents) {
        $params['removeParents'] = $previous_parents;
      }

      $moved_file = $service->files->update($file_id, $updated_file, $params);

      return $this->formatFileResult($moved_file);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to move file in Google Drive: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Formats a Drive file result for consistent output.
   */
  protected function formatFileResult(DriveFile $file): array {
    $formatted = [
      'file_id' => $file->getId(),
      'name' => $file->getName(),
      'mime_type' => $file->getMimeType(),
      'size' => $file->getSize(),
      'created_time' => $file->getCreatedTime(),
      'modified_time' => $file->getModifiedTime(),
      'web_view_link' => $file->getWebViewLink(),
      'web_content_link' => $file->getWebContentLink(),
      'shared' => $file->getShared(),
    ];

    // Add owners if available
    if ($owners = $file->getOwners()) {
      $formatted['owners'] = [];
      foreach ($owners as $owner) {
        $formatted['owners'][] = [
          'display_name' => $owner->getDisplayName(),
          'email_address' => $owner->getEmailAddress(),
        ];
      }
    }

    // Add parents if available
    if ($parents = $file->getParents()) {
      $formatted['parents'] = $parents;
    }

    // Add capabilities if available
    if ($capabilities = $file->getCapabilities()) {
      $formatted['can_edit'] = $capabilities->getCanEdit();
      $formatted['can_share'] = $capabilities->getCanShare();
      $formatted['can_copy'] = $capabilities->getCanCopy();
      $formatted['can_download'] = $capabilities->getCanDownload();
    }

    return $formatted;
  }

  /**
   * Checks if a MIME type is a Google Workspace document.
   */
  protected function isGoogleWorkspaceDocument(string $mime_type): bool {
    return in_array($mime_type, [
      self::MIME_GOOGLE_DOCS,
      self::MIME_GOOGLE_SHEETS,
      self::MIME_GOOGLE_SLIDES,
    ]);
  }

}
