<?php

declare(strict_types=1);

namespace Drupal\eca_google_drive\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Copy files in Google Drive.
 */
#[Action(
  id: 'eca_google_drive_copy_file',
  label: new TranslatableMarkup('Google Drive: Copy File'),
  category: new TranslatableMarkup('Google Drive'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Duplicates files with new names and locations for template duplication, backup creation, and cross-folder copying workflows.'),
  version_introduced: '1.0.0',
)]
final class CopyFile extends DriveActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'source_file_id' => '',
      'new_name' => '',
      'destination_folder_id' => '',
      'copy_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['source_file_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Source File ID'),
      '#description' => $this->t('Google Drive file ID to copy. Supports token replacement.'),
      '#default_value' => $this->configuration['source_file_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['new_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('New Name'),
      '#description' => $this->t('Name for the copied file. If empty, will use original filename with "Copy of" prefix.'),
      '#default_value' => $this->configuration['new_name'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['destination_folder_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Destination Folder ID'),
      '#description' => $this->t('Optional folder ID to copy the file into. Leave empty to copy to same location as original.'),
      '#default_value' => $this->configuration['destination_folder_id'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['copy_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Copy Token Name'),
      '#description' => $this->t('Optional token name to store copied file information. Accessible as [token_name:file_id], [token_name:name], [token_name:web_view_link], etc.'),
      '#default_value' => $this->configuration['copy_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $source_file_id = $this->tokenService->replacePlain($this->configuration['source_file_id']);
    $new_name = $this->tokenService->replacePlain($this->configuration['new_name']);
    $destination_folder_id = $this->tokenService->replacePlain($this->configuration['destination_folder_id']);
    $copy_token_name = $this->tokenService->replacePlain($this->configuration['copy_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('copy file');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($source_file_id)) {
      $this->logger->error('Google Drive copy file action: Missing source file ID.');
      return;
    }


    // If no name provided, get original name and prefix it
    if (empty($new_name)) {
      $original_file = $this->googleDriveService->getFileInfo($auth_type, $client_id, $source_file_id);
      if (!$original_file) {
        $this->logger->error('Google Drive copy file action: Failed to get original file info for naming.');
        return;
      }
      $new_name = 'Copy of ' . $original_file['name'];
    }

    // Build copy parameters
    $copy_params = [
      'name' => $new_name,
    ];

    if (!empty($destination_folder_id)) {
      $copy_params['parent_folder_id'] = $destination_folder_id;
    }

    // Copy the file
    $copied_file_data = $this->googleDriveService->copyFile($auth_type, $client_id, $source_file_id, $copy_params);

    if (!$copied_file_data) {
      $this->logger->error('Google Drive copy file action: Failed to copy file @source_file_id using @auth_type.', [
        '@source_file_id' => $source_file_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store copied file data in token if requested
    if (!empty($copy_token_name)) {
      $this->tokenService->addTokenData($copy_token_name, $copied_file_data);
    }
  }

}