<?php

declare(strict_types=1);

namespace Drupal\eca_google_drive\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Create folders in Google Drive.
 */
#[Action(
  id: 'eca_google_drive_create_folder',
  label: new TranslatableMarkup('Google Drive: Create Folder'),
  category: new TranslatableMarkup('Google Drive'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Creates folder hierarchy programmatically in Google Drive with automatic permission inheritance from parent folders.'),
  version_introduced: '1.0.0',
)]
final class CreateFolder extends DriveActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'folder_name' => '',
      'parent_folder_id' => '',
      'folder_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['folder_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Folder Name'),
      '#description' => $this->t('Name for the new folder. Supports token replacement.'),
      '#default_value' => $this->configuration['folder_name'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['parent_folder_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Parent Folder ID'),
      '#description' => $this->t('Optional parent folder ID to create the folder inside. Leave empty to create in root folder.'),
      '#default_value' => $this->configuration['parent_folder_id'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['folder_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Folder Token Name'),
      '#description' => $this->t('Optional token name to store created folder information. Accessible as [token_name:file_id], [token_name:name], [token_name:web_view_link], etc.'),
      '#default_value' => $this->configuration['folder_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $folder_name = $this->tokenService->replacePlain($this->configuration['folder_name']);
    $parent_folder_id = $this->tokenService->replacePlain($this->configuration['parent_folder_id']);
    $folder_token_name = $this->tokenService->replacePlain($this->configuration['folder_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('create folder');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($folder_name)) {
      $this->logger->error('Google Drive create folder action: Missing folder name.');
      return;
    }


    // Create the folder
    $folder_data = $this->googleDriveService->createFolder($auth_type, $client_id, $folder_name, $parent_folder_id);

    if (!$folder_data) {
      $this->logger->error('Google Drive create folder action: Failed to create folder "@folder_name" using @auth_type.', [
        '@folder_name' => $folder_name,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store folder data in token if requested
    if (!empty($folder_token_name)) {
      $this->tokenService->addTokenData($folder_token_name, $folder_data);
    }
  }

}