<?php

declare(strict_types=1);

namespace Drupal\eca_google_drive\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_google_drive\GoogleDriveService;
use Drupal\file\Entity\File;

/**
 * Download files from Google Drive.
 */
#[Action(
  id: 'eca_google_drive_download_file',
  label: new TranslatableMarkup('Google Drive: Download File'),
  category: new TranslatableMarkup('Google Drive'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Downloads files from Google Drive with format conversion options for Google Workspace documents (PDF, DOCX, XLSX, etc.).'),
  version_introduced: '1.0.0',
)]
final class DownloadFile extends DriveActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'file_id' => '',
      'export_format' => '',
      'save_to_drupal' => '1',
      'local_directory' => 'public://drive-downloads',
      'filename_override' => '',
      'size_limit' => (string) GoogleDriveService::DEFAULT_SIZE_LIMIT,
      'download_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['file_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File ID'),
      '#description' => $this->t('Google Drive file ID to download. Supports token replacement.'),
      '#default_value' => $this->configuration['file_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['export_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Export Format'),
      '#description' => $this->t('Format to export Google Workspace documents (Google Sheets, Docs, Slides only). Regular uploaded files cannot be converted. Leave empty for no conversion.'),
      '#default_value' => $this->configuration['export_format'],
      '#options' => $this->googleDriveService->getExportFormatOptions(),
    ];

    $form['save_to_drupal'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Save to Drupal File System'),
      '#description' => $this->t('Set to "1", "true", or "yes" to save the downloaded file to Drupal\'s file system and create a file entity.'),
      '#default_value' => $this->configuration['save_to_drupal'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['local_directory'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Local Directory'),
      '#description' => $this->t('Directory to save the file. Use Drupal stream wrappers like "public://", "private://", or absolute paths.'),
      '#default_value' => $this->configuration['local_directory'],
      '#eca_token_replacement' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="save_to_drupal"]' => ['filled' => TRUE],
        ],
      ],
    ];

    $form['filename_override'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Filename Override'),
      '#description' => $this->t('Optional custom filename for the downloaded file. If empty, uses original filename.'),
      '#default_value' => $this->configuration['filename_override'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['size_limit'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Size Limit (MB)'),
      '#description' => $this->t('Maximum file size in MB. Default: 50MB. Larger files may cause timeouts.'),
      '#default_value' => $this->configuration['size_limit'],
      '#size' => 10,
    ];

    $form['download_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Download Token Name'),
      '#description' => $this->t('Optional token name to store download information. Accessible as [token_name:content], [token_name:local_path], [token_name:file_entity_id], etc.'),
      '#default_value' => $this->configuration['download_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $file_id = $this->tokenService->replacePlain($this->configuration['file_id']);
    $export_format = $this->configuration['export_format'];
    $save_to_drupal = !empty($this->tokenService->replacePlain($this->configuration['save_to_drupal']));
    $local_directory = $this->tokenService->replacePlain($this->configuration['local_directory']);
    $filename_override = $this->tokenService->replacePlain($this->configuration['filename_override']);
    $size_limit = (int) $this->configuration['size_limit'] * 1048576; // Convert MB to bytes
    $download_token_name = $this->tokenService->replacePlain($this->configuration['download_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('download file');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($file_id)) {
      $this->logger->error('Google Drive download file action: Missing file ID.');
      return;
    }


    // Prepare download parameters
    $download_params = [];
    if (!empty($export_format)) {
      $download_params['export_format'] = $export_format;
    }

    // Download the file
    $download_data = $this->googleDriveService->downloadFile($auth_type, $client_id, $file_id, $download_params);

    if (!$download_data) {
      $this->logger->error('Google Drive download file action: Failed to download file @file_id using @auth_type.', [
        '@file_id' => $file_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Validate file size
    $file_size = strlen($download_data['content']);
    if (!$this->validateFileSize($file_size, $size_limit, 'download file')) {
      return;
    }

    // Determine filename
    $filename = $filename_override ?: $download_data['name'];

    // Add extension based on export format
    if (!empty($export_format) && !empty($download_data['export_format'])) {
      $extension = $this->getExtensionFromMimeType($download_data['export_format']);
      if ($extension && !str_ends_with(strtolower($filename), '.' . $extension)) {
        $filename .= '.' . $extension;
      }
    }

    $result_data = [
      'file_id' => $download_data['file_id'],
      'original_name' => $download_data['name'],
      'filename' => $filename,
      'mime_type' => $download_data['mime_type'],
      'size' => $file_size,
      'content' => $download_data['content'],
      'export_format' => $download_data['export_format'] ?? NULL,
    ];

    // Save to Drupal file system if requested
    if ($save_to_drupal) {
      $directory = $local_directory ?: 'public://drive-downloads';

      // Ensure directory exists
      $file_system = \Drupal::service('file_system');
      $file_system->prepareDirectory($directory, $file_system::CREATE_DIRECTORY | $file_system::MODIFY_PERMISSIONS);

      // Create unique filename if needed
      $file_uri = $file_system->createFilename($filename, $directory);

      // Save file content
      if (file_put_contents($file_uri, $download_data['content'])) {
        // Create Drupal file entity
        $file_entity = File::create([
          'uri' => $file_uri,
          'filename' => $filename,
          'filemime' => $download_data['export_format'] ?: $download_data['mime_type'],
          'status' => 1,
        ]);
        $file_entity->save();

        $result_data['local_path'] = $file_uri;
        $result_data['local_url'] = \Drupal::service('file_url_generator')->generateAbsoluteString($file_uri);
        $result_data['file_entity_id'] = $file_entity->id();
      }
      else {
        $this->logger->error('Google Drive download file action: Failed to save file to local filesystem.');
      }
    }

    // Store download data in token if requested
    if (!empty($download_token_name)) {
      $this->tokenService->addTokenData($download_token_name, $result_data);
    }
  }

  /**
   * Gets file extension from MIME type.
   */
  protected function getExtensionFromMimeType(string $mime_type): ?string {
    $mime_to_ext = [
      GoogleDriveService::EXPORT_PDF => 'pdf',
      GoogleDriveService::EXPORT_DOCX => 'docx',
      GoogleDriveService::EXPORT_XLSX => 'xlsx',
      GoogleDriveService::EXPORT_PPTX => 'pptx',
      GoogleDriveService::EXPORT_CSV => 'csv',
      GoogleDriveService::EXPORT_HTML => 'html',
      GoogleDriveService::EXPORT_TXT => 'txt',
    ];

    return $mime_to_ext[$mime_type] ?? NULL;
  }

}
