<?php

declare(strict_types=1);

namespace Drupal\eca_google_drive\Plugin\Action;

use Drupal\eca_google\Plugin\Action\GoogleApiActionBase;
use Drupal\eca_google_drive\GoogleDriveService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Google Drive ECA actions.
 */
abstract class DriveActionBase extends GoogleApiActionBase {

  /**
   * The Google Drive service.
   */
  protected GoogleDriveService $googleDriveService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->googleDriveService = $container->get('eca_google_drive.google_drive');
    return $instance;
  }


  /**
   * Validates and parses authentication configuration.
   *
   * @param string $auth_client_id
   *   Combined auth_type:client_id string.
   * @param string $action_name
   *   Name of the operation for error logging.
   *
   * @return array|null
   *   Array with 'auth_type' and 'client_id' keys, or NULL on failure.
   */

  /**
   * Validates file size against limits.
   *
   * @param int $file_size
   *   File size in bytes.
   * @param int $size_limit
   *   Size limit in bytes.
   * @param string $operation_name
   *   Name of the operation for error logging.
   *
   * @return bool
   *   TRUE if size is acceptable, FALSE otherwise.
   */
  protected function validateFileSize(int $file_size, int $size_limit, string $operation_name): bool {
    if ($file_size > $size_limit) {
      $this->logger->error('Google Drive @operation action: File size (@size MB) exceeds limit (@limit MB). Consider using background processing for large files.', [
        '@operation' => $operation_name,
        '@size' => round($file_size / 1048576, 2),
        '@limit' => round($size_limit / 1048576, 2),
      ]);
      return FALSE;
    }

    return TRUE;
  }

}
