<?php

declare(strict_types=1);

namespace Drupal\eca_google_drive\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get file information from Google Drive.
 */
#[Action(
  id: 'eca_google_drive_get_file_info',
  label: new TranslatableMarkup('Google Drive: Get File Info'),
  category: new TranslatableMarkup('Google Drive'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieves comprehensive file metadata from Google Drive including size, dates, ownership, sharing status, and capabilities.'),
  version_introduced: '1.0.0',
)]
final class GetFileInfo extends DriveActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'file_id' => '',
      'info_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['file_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File ID'),
      '#description' => $this->t('Google Drive file ID to get information about. Supports token replacement.'),
      '#default_value' => $this->configuration['file_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['info_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File Info Token Name'),
      '#description' => $this->t('Optional token name to store file information. Accessible as [token_name:name], [token_name:size], [token_name:mime_type], [token_name:owners:0:email_address], etc.'),
      '#default_value' => $this->configuration['info_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $file_id = $this->tokenService->replacePlain($this->configuration['file_id']);
    $info_token_name = $this->tokenService->replacePlain($this->configuration['info_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('get file info');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($file_id)) {
      $this->logger->error('Google Drive get file info action: Missing file ID.');
      return;
    }


    // Get file information
    $file_data = $this->googleDriveService->getFileInfo($auth_type, $client_id, $file_id);

    if (!$file_data) {
      $this->logger->error('Google Drive get file info action: Failed to get file info for @file_id using @auth_type.', [
        '@file_id' => $file_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store file data in token if requested
    if (!empty($info_token_name)) {
      $this->tokenService->addTokenData($info_token_name, $file_data);
    }
  }

}
