<?php

declare(strict_types=1);

namespace Drupal\eca_google_drive\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Move files in Google Drive.
 */
#[Action(
  id: 'eca_google_drive_move_file',
  label: new TranslatableMarkup('Google Drive: Move File'),
  category: new TranslatableMarkup('Google Drive'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Relocates files between folders for automated file organization and workflow-based file management with permission inheritance.'),
  version_introduced: '1.0.0',
)]
final class MoveFile extends DriveActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'file_id' => '',
      'destination_folder_id' => '',
      'move_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['file_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File ID'),
      '#description' => $this->t('Google Drive file ID to move. Supports token replacement.'),
      '#default_value' => $this->configuration['file_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['destination_folder_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Destination Folder ID'),
      '#description' => $this->t('Folder ID to move the file into. The file will inherit permissions from the new parent folder. Supports token replacement.'),
      '#default_value' => $this->configuration['destination_folder_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['move_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Move Token Name'),
      '#description' => $this->t('Optional token name to store moved file information. Accessible as [token_name:file_id], [token_name:name], [token_name:parents], etc.'),
      '#default_value' => $this->configuration['move_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $file_id = $this->tokenService->replacePlain($this->configuration['file_id']);
    $destination_folder_id = $this->tokenService->replacePlain($this->configuration['destination_folder_id']);
    $move_token_name = $this->tokenService->replacePlain($this->configuration['move_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('move file');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($file_id)) {
      $this->logger->error('Google Drive move file action: Missing file ID.');
      return;
    }

    if (empty($destination_folder_id)) {
      $this->logger->error('Google Drive move file action: Missing destination folder ID.');
      return;
    }


    // Build move parameters
    $move_params = [
      'parent_folder_id' => $destination_folder_id,
    ];

    // Move the file
    $moved_file_data = $this->googleDriveService->moveFile($auth_type, $client_id, $file_id, $move_params);

    if (!$moved_file_data) {
      $this->logger->error('Google Drive move file action: Failed to move file @file_id to folder @destination_folder_id using @auth_type.', [
        '@file_id' => $file_id,
        '@destination_folder_id' => $destination_folder_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store moved file data in token if requested
    if (!empty($move_token_name)) {
      $this->tokenService->addTokenData($move_token_name, $moved_file_data);
    }
  }

}