<?php

declare(strict_types=1);

namespace Drupal\eca_google_drive\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Search for files in Google Drive.
 */
#[Action(
  id: 'eca_google_drive_search_files',
  label: new TranslatableMarkup('Google Drive: Search Files'),
  category: new TranslatableMarkup('Google Drive'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Advanced file search in Google Drive with query syntax support for filtering by name, type, date, owner, and more.'),
  version_introduced: '1.0.0',
)]
final class SearchFiles extends DriveActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'search_query' => '',
      'max_results' => '10',
      'page_token' => '',
      'search_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['search_query'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search Query'),
      '#description' => $this->t('Google Drive search query. Examples: "name contains \'report\'", "mimeType=\'application/pdf\'", "createdTime > \'2024-01-01\'", "parents in \'folder_id\'". Leave empty to list all accessible files.'),
      '#default_value' => $this->configuration['search_query'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['max_results'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Results'),
      '#description' => $this->t('Maximum number of files to return (1-1000). Default: 10.'),
      '#default_value' => $this->configuration['max_results'],
      '#min' => 1,
      '#max' => 1000,
    ];

    $form['page_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Page Token'),
      '#description' => $this->t('Token for pagination. Use next_page_token from previous search results to get more files.'),
      '#default_value' => $this->configuration['page_token'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['search_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search Results Token Name'),
      '#description' => $this->t('Optional token name to store search results. Accessible as [token_name:files:0:name], [token_name:total_files], [token_name:next_page_token], etc.'),
      '#default_value' => $this->configuration['search_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['examples_list'] = [
      '#type' => 'item',
      '#title' => $this->t('Query Examples'),
      '#markup' => $this->t('
        <ul>
          <li><strong>name contains "report"</strong> - Files with "report" in the name</li>
          <li><strong>mimeType="application/pdf"</strong> - PDF files only</li>
          <li><strong>createdTime > "2024-01-01"</strong> - Files created after Jan 1, 2024</li>
          <li><strong>parents in "folder_id"</strong> - Files in specific folder</li>
          <li><strong>starred = true</strong> - Starred files</li>
          <li><strong>trashed = false</strong> - Non-trashed files</li>
          <li><strong>shared = true</strong> - Shared files</li>
          <li><strong>"me" in owners</strong> - Files owned by authenticated user</li>
        </ul>
      '),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $search_query = $this->tokenService->replacePlain($this->configuration['search_query']);
    $max_results = (int) $this->configuration['max_results'];
    $page_token = $this->tokenService->replacePlain($this->configuration['page_token'], [], ['clear' => TRUE]);
    $search_token_name = $this->tokenService->replacePlain($this->configuration['search_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('search files');
    if (!$auth_type || !$client_id) {
      return;
    }


    // Build search parameters
    $search_params = [
      'max_results' => min($max_results, 1000),
    ];

    if (!empty($search_query)) {
      $search_params['query'] = $search_query;
    }

    if (!empty($page_token)) {
      $search_params['page_token'] = $page_token;
    }

    // Search for files
    $search_results = $this->googleDriveService->searchFiles($auth_type, $client_id, $search_params);

    if (!$search_results) {
      $this->logger->error('Google Drive search files action: Failed to search files using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store search results in token if requested
    if (!empty($search_token_name)) {
      $this->tokenService->addTokenData($search_token_name, $search_results);
    }
  }

}
