<?php

declare(strict_types=1);

namespace Drupal\eca_google_drive\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_google_drive\GoogleDriveService;

/**
 * Share files or folders in Google Drive.
 */
#[Action(
  id: 'eca_google_drive_share_file',
  label: new TranslatableMarkup('Google Drive: Share File/Folder'),
  category: new TranslatableMarkup('Google Drive'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Grants permissions to users, groups, or domains for Google Drive files and folders with configurable notification options.'),
  version_introduced: '1.0.0',
)]
final class ShareFile extends DriveActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'file_id' => '',
      'permission_type' => GoogleDriveService::TYPE_USER,
      'permission_role' => GoogleDriveService::ROLE_READER,
      'email_address' => '',
      'domain' => '',
      'send_notification' => '1',
      'notification_message' => '',
      'permission_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['file_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File/Folder ID'),
      '#description' => $this->t('Google Drive file or folder ID to share. Supports token replacement.'),
      '#default_value' => $this->configuration['file_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['permission_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Permission Type'),
      '#description' => $this->t('Type of recipient for the permission.'),
      '#default_value' => $this->configuration['permission_type'],
      '#options' => $this->googleDriveService->getPermissionTypeOptions(),
      '#required' => TRUE,
    ];

    $form['permission_role'] = [
      '#type' => 'select',
      '#title' => $this->t('Permission Role'),
      '#description' => $this->t('Level of access to grant.'),
      '#default_value' => $this->configuration['permission_role'],
      '#options' => $this->googleDriveService->getPermissionRoleOptions(),
      '#required' => TRUE,
    ];

    $form['email_address'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Email Address'),
      '#description' => $this->t('Email address for user or group permissions. Supports token replacement.'),
      '#default_value' => $this->configuration['email_address'],
      '#eca_token_replacement' => TRUE,
      '#states' => [
        'visible' => [
          [':input[name="permission_type"]' => ['value' => GoogleDriveService::TYPE_USER]],
          'or',
          [':input[name="permission_type"]' => ['value' => GoogleDriveService::TYPE_GROUP]],
        ],
        'required' => [
          [':input[name="permission_type"]' => ['value' => GoogleDriveService::TYPE_USER]],
          'or',
          [':input[name="permission_type"]' => ['value' => GoogleDriveService::TYPE_GROUP]],
        ],
      ],
    ];

    $form['domain'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Domain'),
      '#description' => $this->t('Domain name for domain-wide permissions (e.g., "example.com"). Supports token replacement.'),
      '#default_value' => $this->configuration['domain'],
      '#eca_token_replacement' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="permission_type"]' => ['value' => GoogleDriveService::TYPE_DOMAIN],
        ],
        'required' => [
          ':input[name="permission_type"]' => ['value' => GoogleDriveService::TYPE_DOMAIN],
        ],
      ],
    ];

    $form['send_notification'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Send Notification'),
      '#description' => $this->t('Set to "1", "true", or "yes" to send email notification to the recipient. Set to "0", "false", or "no" to skip notification.'),
      '#default_value' => $this->configuration['send_notification'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['notification_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Notification Message'),
      '#description' => $this->t('Optional custom message to include in the notification email. Supports token replacement.'),
      '#default_value' => $this->configuration['notification_message'],
      '#eca_token_replacement' => TRUE,
      '#rows' => 3,
    ];

    $form['permission_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Permission Token Name'),
      '#description' => $this->t('Optional token name to store permission information. Accessible as [token_name:permission_id], [token_name:role], [token_name:email_address], etc.'),
      '#default_value' => $this->configuration['permission_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $file_id = $this->tokenService->replacePlain($this->configuration['file_id']);
    $permission_type = $this->configuration['permission_type'];
    $permission_role = $this->configuration['permission_role'];
    $email_address = $this->tokenService->replacePlain($this->configuration['email_address']);
    $domain = $this->tokenService->replacePlain($this->configuration['domain']);
    $send_notification = !empty($this->tokenService->replacePlain($this->configuration['send_notification'], [], ['clear' => TRUE]));
    $notification_message = $this->tokenService->replace($this->configuration['notification_message'], [], ['clear' => TRUE]);
    $permission_token_name = $this->tokenService->replacePlain($this->configuration['permission_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('share file');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($file_id)) {
      $this->logger->error('Google Drive share file action: Missing file/folder ID.');
      return;
    }

    // Validate required fields based on permission type
    if (in_array($permission_type, [GoogleDriveService::TYPE_USER, GoogleDriveService::TYPE_GROUP]) && empty($email_address)) {
      $this->logger->error('Google Drive share file action: Email address is required for user and group permissions.');
      return;
    }

    if ($permission_type === GoogleDriveService::TYPE_DOMAIN && empty($domain)) {
      $this->logger->error('Google Drive share file action: Domain is required for domain permissions.');
      return;
    }


    // Build permission parameters
    $permission_params = [
      'type' => $permission_type,
      'role' => $permission_role,
      'send_notification' => $send_notification,
    ];

    if (!empty($email_address)) {
      $permission_params['email_address'] = $email_address;
    }

    if (!empty($domain)) {
      $permission_params['domain'] = $domain;
    }

    if (!empty($notification_message)) {
      $permission_params['message'] = $notification_message;
    }

    // Share the file/folder
    $permission_data = $this->googleDriveService->shareFile($auth_type, $client_id, $file_id, $permission_params);

    if (!$permission_data) {
      $this->logger->error('Google Drive share file action: Failed to share file @file_id using @auth_type.', [
        '@file_id' => $file_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store permission data in token if requested
    if (!empty($permission_token_name)) {
      $this->tokenService->addTokenData($permission_token_name, $permission_data);
    }
  }

}
