<?php

declare(strict_types=1);

namespace Drupal\eca_google_drive\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_google_drive\GoogleDriveService;
use Drupal\file\FileInterface;

/**
 * Upload files to Google Drive.
 */
#[Action(
  id: 'eca_google_drive_upload_file',
  label: new TranslatableMarkup('Google Drive: Upload File'),
  category: new TranslatableMarkup('Google Drive'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Uploads files from URLs, file fields, or generated content to Google Drive with support for various file sources and formats.'),
  version_introduced: '1.0.0',
)]
final class UploadFile extends DriveActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'source_type' => 'url',
      'source_url' => '',
      'source_file_field' => '',
      'source_content' => '',
      'file_name' => '',
      'parent_folder_id' => '',
      'size_limit' => (string) GoogleDriveService::DEFAULT_SIZE_LIMIT,
      'file_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['source_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Source Type'),
      '#description' => $this->t('Choose how to provide the file to upload.'),
      '#default_value' => $this->configuration['source_type'],
      '#options' => [
        'url' => $this->t('URL'),
        'file_field' => $this->t('File Field/Entity'),
        'content' => $this->t('Generated Content'),
      ],
      '#required' => TRUE,
    ];

    $form['source_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Source URL'),
      '#description' => $this->t('URL of the file to upload. Supports token replacement.'),
      '#default_value' => $this->configuration['source_url'],
      '#eca_token_replacement' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="source_type"]' => ['value' => 'url'],
        ],
      ],
    ];

    $form['source_file_field'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File Field Token'),
      '#description' => $this->t('Token representing a file field or file entity. Examples: [entity:field_file], [entity:field_file:entity:path], [entity:field_file:entity:url].'),
      '#default_value' => $this->configuration['source_file_field'],
      '#eca_token_replacement' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="source_type"]' => ['value' => 'file_field'],
        ],
      ],
    ];

    $form['source_content'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Generated Content'),
      '#description' => $this->t('Raw content to upload as a file. Supports token replacement for dynamic content generation.'),
      '#default_value' => $this->configuration['source_content'],
      '#eca_token_replacement' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="source_type"]' => ['value' => 'content'],
        ],
      ],
    ];

    $form['file_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File Name'),
      '#description' => $this->t('Name for the uploaded file in Google Drive. If empty, will use original filename or generate one.'),
      '#default_value' => $this->configuration['file_name'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['parent_folder_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Parent Folder ID'),
      '#description' => $this->t('Optional Google Drive folder ID to upload the file into. Leave empty to upload to root folder.'),
      '#default_value' => $this->configuration['parent_folder_id'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['size_limit'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Size Limit (MB)'),
      '#description' => $this->t('Maximum file size in MB. Default: 50MB. Larger files may cause timeouts.'),
      '#default_value' => $this->configuration['size_limit'],
      '#size' => 10,
    ];

    $form['file_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File Token Name'),
      '#description' => $this->t('Optional token name to store uploaded file information. Accessible as [token_name:file_id], [token_name:name], [token_name:web_view_link], etc.'),
      '#default_value' => $this->configuration['file_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $source_type = $this->configuration['source_type'];
    $source_url = $this->tokenService->replacePlain($this->configuration['source_url']);
    $source_file_field = $this->tokenService->replace($this->configuration['source_file_field']);
    $source_content = $this->tokenService->replace($this->configuration['source_content']);
    $file_name = $this->tokenService->replacePlain($this->configuration['file_name']);
    $parent_folder_id = $this->tokenService->replacePlain($this->configuration['parent_folder_id']);
    $size_limit = (int) $this->configuration['size_limit'] * 1048576; // Convert MB to bytes
    $file_token_name = $this->tokenService->replacePlain($this->configuration['file_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('upload file');
    if (!$auth_type || !$client_id) {
      return;
    }


    // Prepare upload parameters
    $upload_params = [
      'size_limit' => $size_limit,
    ];

    if (!empty($parent_folder_id)) {
      $upload_params['parent_folder_id'] = $parent_folder_id;
    }

    // Handle different source types
    switch ($source_type) {
      case 'url':
        if (empty($source_url)) {
          $this->logger->error('Google Drive upload file action: Missing source URL.');
          return;
        }

        $upload_params['url'] = $source_url;
        $upload_params['name'] = $file_name ?: basename($source_url);

        // Get file size from headers
        $headers = get_headers($source_url, TRUE);
        $upload_params['file_size'] = isset($headers['Content-Length']) ? (int) $headers['Content-Length'] : 0;
        $upload_params['mime_type'] = $headers['Content-Type'] ?? 'application/octet-stream';
        break;

      case 'file_field':
        if ($source_file_field instanceof FileInterface) {
          $file_uri = $source_file_field->getFileUri();
          $upload_params['file_path'] = \Drupal::service('file_system')->realpath($file_uri);
          $upload_params['name'] = $file_name ?: $source_file_field->getFilename();
          $upload_params['file_size'] = $source_file_field->getSize();
          $upload_params['mime_type'] = $source_file_field->getMimeType();
        }
        elseif (is_string($source_file_field)) {
          if (empty($source_file_field)) {
            $this->logger->error('Google Drive upload file action: Empty file path from token.');
            return;
          }

          // Convert relative URL to absolute file path
          $file_system = \Drupal::service('file_system');
          $real_path = FALSE;

          // Handle URLs that include web root path like "/connector/web/sites/default/files/..."
          $relative_path = ltrim($source_file_field, '/');

          // Look for sites/default/files pattern anywhere in the path
          if (preg_match('#(.*/)?sites/default/files/(.+)#', $relative_path, $matches)) {
            $filename = $matches[2];
            $file_uri = 'public://' . $filename;
            $real_path = $file_system->realpath($file_uri);
          }
          // Check if it's a stream wrapper URI like "public://file.csv"
          elseif (preg_match('#^[a-zA-Z][a-zA-Z0-9+.-]*://#', $source_file_field)) {
            $real_path = $file_system->realpath($source_file_field);
          }
          // Try as absolute path directly
          elseif (file_exists($source_file_field)) {
            $real_path = $source_file_field;
          }
          // If it's just a filename with no path, we can't locate it
          else {
            $this->logger->error('Google Drive upload file action: Cannot locate file. Token returned filename without path or invalid path: @path. Try using [entity:field_file] instead of [entity:field_file:entity].', [
              '@path' => $source_file_field,
            ]);
            return;
          }

          if ($real_path && file_exists($real_path)) {
            $upload_params['file_path'] = $real_path;
            $upload_params['name'] = $file_name ?: basename($source_file_field);
            $upload_params['file_size'] = filesize($real_path);
            $upload_params['mime_type'] = mime_content_type($real_path) ?: 'application/octet-stream';
          }
          else {
            $this->logger->error('Google Drive upload file action: File not found or invalid path: @path', [
              '@path' => $source_file_field,
            ]);
            return;
          }
        }
        else {
          $this->logger->error('Google Drive upload file action: Invalid file field or file not found.');
          return;
        }
        break;

      case 'content':
        if (empty($source_content)) {
          $this->logger->error('Google Drive upload file action: Missing source content.');
          return;
        }

        $upload_params['file_content'] = $source_content;
        $upload_params['name'] = $file_name ?: 'generated_file.txt';
        $upload_params['file_size'] = strlen($source_content);
        $upload_params['mime_type'] = 'text/plain';
        break;

      default:
        $this->logger->error('Google Drive upload file action: Invalid source type.');
        return;
    }

    // Validate file size
    if (!$this->validateFileSize($upload_params['file_size'], $upload_params['size_limit'], 'upload file')) {
      return;
    }

    // Upload the file
    $file_data = $this->googleDriveService->uploadFile($auth_type, $client_id, $upload_params);

    if (!$file_data) {
      $this->logger->error('Google Drive upload file action: Failed to upload file using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store file data in token if requested
    if (!empty($file_token_name)) {
      $this->tokenService->addTokenData($file_token_name, $file_data);
    }
  }

}
