<?php

declare(strict_types=1);

namespace Drupal\eca_google_meet;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\eca_google\GoogleApiService;
use Google\Service\Meet;
use Google\Service\Meet\Space;

/**
 * Service for Google Meet API operations.
 */
class GoogleMeetService {

  use StringTranslationTrait;

  public const string ACCESS_TYPE_OPEN = 'OPEN';
  public const string ACCESS_TYPE_TRUSTED = 'TRUSTED';
  public const string ACCESS_TYPE_RESTRICTED = 'RESTRICTED';

  public const string ENTRY_POINT_ALL = 'ALL';

  /**
   * The Google API service.
   */
  protected GoogleApiService $googleApiService;

  /**
   * The logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * @param GoogleApiService $google_api_service
   * @param LoggerChannelFactoryInterface $logger_factory
   */
  public function __construct(
    GoogleApiService $google_api_service,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->googleApiService = $google_api_service;
    $this->logger = $logger_factory->get('eca_google_meet');
  }

  /**
   * Gets a configured Google Meet service instance.
   *
   * @param string $auth_type
   *   The authentication type ('api_client' or 'service_account').
   * @param string $client_id
   *   The Google API client entity ID.
   *
   * @return \Google\Service\Meet|null
   *   The Google Meet service instance or NULL on failure.
   */
  public function getMeetService(string $auth_type, string $client_id): ?Meet {
    $service = $this->googleApiService->getService('meet', $auth_type, $client_id);
    return $service instanceof Meet ? $service : NULL;
  }

  /**
   * Validates Google Meet API access.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   *
   * @return bool
   *   TRUE if Meet API is accessible, FALSE otherwise.
   */
  public function validateMeetAccess(string $auth_type, string $client_id): bool {
    return $this->googleApiService->validateApiAccess('meet', $auth_type, $client_id);
  }

  /**
   * Gets access type options for meeting spaces.
   *
   * @return array
   *   Array of access type options.
   */
  public function getAccessTypeOptions(): array {
    return [
      self::ACCESS_TYPE_OPEN => $this->t('Open - Anyone can join'),
      self::ACCESS_TYPE_TRUSTED => $this->t('Trusted - Organization members and invited users'),
      self::ACCESS_TYPE_RESTRICTED => $this->t('Restricted - Only invited participants'),
    ];
  }


  /**
   * Creates a new Google Meet space.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param array $config
   *   Space configuration options.
   *
   * @return array|null
   *   Space details or NULL on failure.
   */
  public function createMeetingSpace(string $auth_type, string $client_id, array $config = []): ?array {
    $service = $this->getMeetService($auth_type, $client_id);

    if (!$service) {
      return NULL;
    }

    try {
      $space = new \Google\Service\Meet\Space();

      if (!empty($config)) {
        $spaceConfig = new \Google\Service\Meet\SpaceConfig();

        if (isset($config['access_type'])) {
          $spaceConfig->setAccessType($config['access_type']);
        }

        if (isset($config['entry_point_access'])) {
          $spaceConfig->setEntryPointAccess($config['entry_point_access']);
        }

        $space->setConfig($spaceConfig);
      }

      /** @var Space $result */
      $result = $service->spaces->create($space);

      return [
        'space_id' => $result->getName(),
        'meeting_uri' => $result->getMeetingUri(),
        'meeting_code' => $result->getMeetingCode(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create Google Meet space: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Lists conference records with optional filtering.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param array $filters
   *   Optional filters (date range, etc.).
   *
   * @return array|null
   *   Array of conference records or NULL on failure.
   */
  public function listConferenceRecords(string $auth_type, string $client_id, array $filters = []): ?array {
    $service = $this->getMeetService($auth_type, $client_id);
    if (!$service) {
      return NULL;
    }

    try {
      $options = [];

      if (!empty($filters['filter'])) {
        $options['filter'] = $filters['filter'];
      }

      if (!empty($filters['page_size'])) {
        $options['pageSize'] = $filters['page_size'];
      }

      $result = $service->conferenceRecords->listConferenceRecords($options);
      $records = [];

      foreach ($result->getConferenceRecords() as $record) {
        $records[] = [
          'name' => $record->getName(),
          'start_time' => $record->getStartTime(),
          'end_time' => $record->getEndTime(),
          'space' => $record->getSpace(),
        ];
      }

      return $records;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to list Google Meet conference records: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets a specific conference record with detailed metadata.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param string $conference_record_name
   *   The conference record name.
   *
   * @return array|null
   *   Conference record details or NULL on failure.
   */
  public function getConferenceRecord(string $auth_type, string $client_id, string $conference_record_name): ?array {
    $service = $this->getMeetService($auth_type, $client_id);
    if (!$service) {
      return NULL;
    }

    try {
      $record = $service->conferenceRecords->get($conference_record_name);
      
      return [
        'name' => $record->getName(),
        'start_time' => $record->getStartTime(),
        'end_time' => $record->getEndTime(),
        'space' => $record->getSpace(),
        'expire_time' => $record->getExpireTime(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get Google Meet conference record: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Updates a Google Meet space configuration.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param string $space_name
   *   The space name/ID.
   * @param array $config
   *   New configuration options.
   *
   * @return array|null
   *   Updated space details or NULL on failure.
   */
  public function updateMeetingSpace(string $auth_type, string $client_id, string $space_name, array $config): ?array {
    $service = $this->getMeetService($auth_type, $client_id);
    if (!$service) {
      return NULL;
    }

    try {
      $space = new \Google\Service\Meet\Space();
      $spaceConfig = new \Google\Service\Meet\SpaceConfig();
      
      if (isset($config['access_type'])) {
        $spaceConfig->setAccessType($config['access_type']);
      }
      
      $space->setConfig($spaceConfig);

      $result = $service->spaces->patch($space_name, $space, ['updateMask' => 'config.access_type']);
      
      return [
        'space_id' => $result->getName(),
        'meeting_uri' => $result->getMeetingUri(),
        'meeting_code' => $result->getMeetingCode(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to update Google Meet space: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Lists recordings for a conference record.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param string $conference_record_name
   *   The conference record name.
   *
   * @return array|null
   *   Array of recordings or NULL on failure.
   */
  public function listRecordings(string $auth_type, string $client_id, string $conference_record_name): ?array {
    $service = $this->getMeetService($auth_type, $client_id);
    if (!$service) {
      return NULL;
    }

    try {
      $result = $service->conferenceRecords_recordings->listConferenceRecordsRecordings($conference_record_name);
      $recordings = [];

      foreach ($result->getRecordings() as $recording) {
        $recordings[] = [
          'name' => $recording->getName(),
          'drive_destination' => $recording->getDriveDestination(),
          'start_time' => $recording->getStartTime(),
          'end_time' => $recording->getEndTime(),
          'state' => $recording->getState(),
        ];
      }

      return $recordings;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to list Google Meet recordings: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Lists transcripts for a conference record.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param string $conference_record_name
   *   The conference record name.
   *
   * @return array|null
   *   Array of transcripts or NULL on failure.
   */
  public function listTranscripts(string $auth_type, string $client_id, string $conference_record_name): ?array {
    $service = $this->getMeetService($auth_type, $client_id);
    if (!$service) {
      return NULL;
    }

    try {
      $result = $service->conferenceRecords_transcripts->listConferenceRecordsTranscripts($conference_record_name);
      $transcripts = [];

      foreach ($result->getTranscripts() as $transcript) {
        $transcripts[] = [
          'name' => $transcript->getName(),
          'docs_destination' => $transcript->getDocsDestination(),
          'start_time' => $transcript->getStartTime(),
          'end_time' => $transcript->getEndTime(),
          'state' => $transcript->getState(),
        ];
      }

      return $transcripts;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to list Google Meet transcripts: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Lists participants for a conference record.
   *
   * @param string $auth_type
   *   The authentication type.
   * @param string $client_id
   *   The client ID.
   * @param string $conference_record_name
   *   The conference record name.
   *
   * @return array|null
   *   Array of participants or NULL on failure.
   */
  public function listParticipants(string $auth_type, string $client_id, string $conference_record_name): ?array {
    $service = $this->getMeetService($auth_type, $client_id);
    if (!$service) {
      return NULL;
    }

    try {
      $result = $service->conferenceRecords_participants->listConferenceRecordsParticipants($conference_record_name);
      $participants = [];

      foreach ($result->getParticipants() as $participant) {
        $participants[] = [
          'name' => $participant->getName(),
          'earliest_start_time' => $participant->getEarliestStartTime(),
          'latest_end_time' => $participant->getLatestEndTime(),
          'anonymous_user' => $participant->getAnonymousUser(),
          'signed_in_user' => $participant->getSignedInUser(),
          'phone_user' => $participant->getPhoneUser(),
        ];
      }

      return $participants;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to list Google Meet participants: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

}
