<?php

declare(strict_types=1);

namespace Drupal\eca_google_meet\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get detailed information about a Google Meet conference record.
 */
#[Action(
  id: 'eca_google_meet_get_record',
  label: new TranslatableMarkup('Google Meet: Get Conference Record'),
  category: new TranslatableMarkup('Google Meet'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieves detailed metadata for a specific Google Meet conference record.'),
  version_introduced: '1.0.0',
)]
final class GetConferenceRecord extends MeetActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'conference_record_name' => '',
      'token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['conference_record_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Conference Record Name'),
      '#description' => $this->t('The conference record identifier (e.g., "conferenceRecords/abc123"). Often obtained from List Conference Records action. Supports token replacement.'),
      '#default_value' => $this->configuration['conference_record_name'],
      '#eca_token_replacement' => TRUE,
      '#required' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Record Token Name'),
      '#description' => $this->t('Token name to store conference record details. Accessible as [token_name:name], [token_name:start_time], [token_name:end_time], [token_name:expire_time], etc.'),
      '#default_value' => $this->configuration['token_name'],
      '#required' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $conference_record_name = $this->tokenService->replacePlain($this->configuration['conference_record_name']);
    $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);


    [$auth_type, $client_id] = $this->validateAndParseAuth('meet operation');
    if (!$auth_type || !$client_id) {
      return;
    }


    $record = $this->googleMeetService->getConferenceRecord($auth_type, $client_id, $conference_record_name);

    if ($record === NULL) {
      $this->logger->error('Google Meet get record action: Failed to retrieve conference record @conference using @auth_type.', [
        '@conference' => $conference_record_name,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    $this->tokenService->addTokenData($token_name, $record);
  }


}
