<?php

declare(strict_types=1);

namespace Drupal\eca_google_meet\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * List Google Meet conference records.
 */
#[Action(
  id: 'eca_google_meet_list_records',
  label: new TranslatableMarkup('Google Meet: List Conference Records'),
  category: new TranslatableMarkup('Google Meet'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieves historical Google Meet conference records with optional filtering.'),
  version_introduced: '1.0.0',
)]
final class ListConferenceRecords extends MeetActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'filter_expression' => '',
      'page_size' => '10',
      'token_name' => '',
      'count_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['filter_expression'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Filter Expression'),
      '#description' => $this->t('Optional filter to limit results. Example: "start_time > 2024-01-01T00:00:00Z". Leave empty to list all records. Supports token replacement.'),
      '#default_value' => $this->configuration['filter_expression'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['page_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Page Size'),
      '#description' => $this->t('Maximum number of records to return (1-100).'),
      '#default_value' => $this->configuration['page_size'],
      '#min' => 1,
      '#max' => 100,
      '#required' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Records Token Name'),
      '#description' => $this->t('Token name to store conference records. Data accessible as [token_name:0:name], [token_name:1:start_time], etc.'),
      '#default_value' => $this->configuration['token_name'],
      '#required' => TRUE,
    ];

    $form['count_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Count Token Name'),
      '#description' => $this->t('Optional token name to store the number of conference records found.'),
      '#default_value' => $this->configuration['count_token_name'],
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $filter_expression = $this->tokenService->replacePlain($this->configuration['filter_expression']);
    $page_size = (int) $this->configuration['page_size'];
    $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);
    $count_token_name = $this->tokenService->replacePlain($this->configuration['count_token_name']);


    [$auth_type, $client_id] = $this->validateAndParseAuth('meet operation');
    if (!$auth_type || !$client_id) {
      return;
    }


    $filters = ['page_size' => $page_size];
    if (!empty($filter_expression)) {
      $filters['filter'] = $filter_expression;
    }

    $records = $this->googleMeetService->listConferenceRecords($auth_type, $client_id, $filters);

    if ($records === NULL) {
      $this->logger->error('Google Meet list records action: Failed to retrieve conference records using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    $formatted_data = [];
    foreach ($records as $index => $record) {
      $formatted_data[$index] = [
        'name' => $record['name'] ?? '',
        'start_time' => $record['start_time'] ?? '',
        'end_time' => $record['end_time'] ?? '',
        'space' => $record['space'] ?? '',
      ];
    }

    $this->tokenService->addTokenData($token_name, $formatted_data);

    if (!empty($count_token_name)) {
      $this->tokenService->addTokenData($count_token_name, count($records));
    }
  }


}
