<?php

declare(strict_types=1);

namespace Drupal\eca_google_meet\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * List recordings for a Google Meet conference record.
 */
#[Action(
  id: 'eca_google_meet_list_recordings',
  label: new TranslatableMarkup('Google Meet: List Recordings'),
  category: new TranslatableMarkup('Google Meet'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Lists available recordings for a Google Meet conference with metadata and download information.'),
  version_introduced: '1.0.0',
)]
final class ListRecordings extends MeetActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'conference_record_name' => '',
      'token_name' => '',
      'count_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['conference_record_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Conference Record Name'),
      '#description' => $this->t('The conference record identifier (e.g., "conferenceRecords/abc123"). Often obtained from List Conference Records action. Supports token replacement.'),
      '#default_value' => $this->configuration['conference_record_name'],
      '#eca_token_replacement' => TRUE,
      '#required' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Recordings Token Name'),
      '#description' => $this->t('Token name to store recording data. Accessible as [token_name:0:name], [token_name:0:drive_destination], [token_name:0:state], etc.'),
      '#default_value' => $this->configuration['token_name'],
      '#required' => TRUE,
    ];

    $form['count_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Count Token Name'),
      '#description' => $this->t('Optional token name to store the number of recordings found.'),
      '#default_value' => $this->configuration['count_token_name'],
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $conference_record_name = $this->tokenService->replacePlain($this->configuration['conference_record_name']);
    $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);
    $count_token_name = $this->tokenService->replacePlain($this->configuration['count_token_name']);


    [$auth_type, $client_id] = $this->validateAndParseAuth('meet operation');
    if (!$auth_type || !$client_id) {
      return;
    }


    $recordings = $this->googleMeetService->listRecordings($auth_type, $client_id, $conference_record_name);

    if ($recordings === NULL) {
      $this->logger->error('Google Meet list recordings action: Failed to retrieve recordings for @conference using @auth_type.', [
        '@conference' => $conference_record_name,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    $formatted_data = [];
    foreach ($recordings as $index => $recording) {
      $formatted_data[$index] = [
        'name' => $recording['name'] ?? '',
        'drive_destination' => $recording['drive_destination'] ?? '',
        'start_time' => $recording['start_time'] ?? '',
        'end_time' => $recording['end_time'] ?? '',
        'state' => $recording['state'] ?? '',
      ];
    }

    $this->tokenService->addTokenData($token_name, $formatted_data);

    if (!empty($count_token_name)) {
      $this->tokenService->addTokenData($count_token_name, count($recordings));
    }
  }


}
