<?php

declare(strict_types=1);

namespace Drupal\eca_google_meet\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * List transcripts for a Google Meet conference record.
 */
#[Action(
  id: 'eca_google_meet_list_transcripts',
  label: new TranslatableMarkup('Google Meet: List Transcripts'),
  category: new TranslatableMarkup('Google Meet'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Lists available transcripts for a Google Meet conference with metadata and document links.'),
  version_introduced: '1.0.0',
)]
final class ListTranscripts extends MeetActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'conference_record_name' => '',
      'token_name' => '',
      'count_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['conference_record_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Conference Record Name'),
      '#description' => $this->t('The conference record identifier (e.g., "conferenceRecords/abc123"). Often obtained from List Conference Records action. Supports token replacement.'),
      '#default_value' => $this->configuration['conference_record_name'],
      '#eca_token_replacement' => TRUE,
      '#required' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Transcripts Token Name'),
      '#description' => $this->t('Token name to store transcript data. Accessible as [token_name:0:name], [token_name:0:docs_destination], [token_name:0:state], etc.'),
      '#default_value' => $this->configuration['token_name'],
      '#required' => TRUE,
    ];

    $form['count_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Count Token Name'),
      '#description' => $this->t('Optional token name to store the number of transcripts found.'),
      '#default_value' => $this->configuration['count_token_name'],
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $conference_record_name = $this->tokenService->replacePlain($this->configuration['conference_record_name']);
    $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);
    $count_token_name = $this->tokenService->replacePlain($this->configuration['count_token_name']);


    [$auth_type, $client_id] = $this->validateAndParseAuth('meet operation');
    if (!$auth_type || !$client_id) {
      return;
    }


    $transcripts = $this->googleMeetService->listTranscripts($auth_type, $client_id, $conference_record_name);

    if ($transcripts === NULL) {
      $this->logger->error('Google Meet list transcripts action: Failed to retrieve transcripts for @conference using @auth_type.', [
        '@conference' => $conference_record_name,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    $formatted_data = [];
    foreach ($transcripts as $index => $transcript) {
      $formatted_data[$index] = [
        'name' => $transcript['name'] ?? '',
        'docs_destination' => $transcript['docs_destination'] ?? '',
        'start_time' => $transcript['start_time'] ?? '',
        'end_time' => $transcript['end_time'] ?? '',
        'state' => $transcript['state'] ?? '',
      ];
    }

    $this->tokenService->addTokenData($token_name, $formatted_data);

    if (!empty($count_token_name)) {
      $this->tokenService->addTokenData($count_token_name, count($transcripts));
    }
  }


}
