<?php

declare(strict_types=1);

namespace Drupal\eca_google_meet\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca_google\Plugin\Action\GoogleApiActionBase;
use Drupal\eca_google_meet\GoogleMeetService;
use Symfony\Component\DependencyInjection\ContainerInterface;

abstract class MeetActionBase extends GoogleApiActionBase {

  /**
   * The Google Meet service.
   */
  protected GoogleMeetService $googleMeetService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->googleMeetService = $container->get('eca_google_meet.google_meet');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);
    $this->validateMeetAccess($form, $form_state);
  }


  protected function validateMeetAccess(array $form, FormStateInterface $form_state): void {
    $auth_client_id = $form_state->getValue('auth_client_id');
    if (!empty($auth_client_id)) {
      [$auth_type, $client_id] = explode(':', $auth_client_id, 2);

      if (!in_array($auth_type, ['api_client', 'service_account'])) {
        $form_state->setErrorByName('auth_client_id', $this->t('Auth type must be "api_client" or "service_account".'));
        return;
      }

      if (!$this->googleMeetService->validateMeetAccess($auth_type, $client_id)) {
        $form_state->setErrorByName('auth_client_id', $this->t('The selected Google API client does not have access to Google Meet API.'));
      }
    }
  }


}
