<?php

declare(strict_types=1);

namespace Drupal\eca_google_meet\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_google_meet\GoogleMeetService;

/**
 * Update a Google Meet space configuration.
 */
#[Action(
  id: 'eca_google_meet_update_space',
  label: new TranslatableMarkup('Google Meet: Update Meeting Space'),
  category: new TranslatableMarkup('Google Meet'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Updates access control settings for an existing Google Meet space.'),
  version_introduced: '1.0.0',
)]
final class UpdateMeetingSpace extends MeetActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'space_name' => '',
      'access_type' => GoogleMeetService::ACCESS_TYPE_TRUSTED,
      'space_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['space_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Space Name'),
      '#description' => $this->t('The Google Meet space identifier (e.g., "spaces/abc123"). Often obtained from Create Meeting Space action. Supports token replacement.'),
      '#default_value' => $this->configuration['space_name'],
      '#eca_token_replacement' => TRUE,
      '#required' => TRUE,
    ];

    $form['access_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Access Type'),
      '#description' => $this->t('New access control setting for the meeting space. RESTRICTED requires participants to be invited via Google Calendar events.'),
      '#options' => $this->googleMeetService->getAccessTypeOptions(),
      '#default_value' => $this->configuration['access_type'],
      '#required' => TRUE,
    ];

    $form['space_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Space Token Name'),
      '#description' => $this->t('Optional token name to store updated meeting space details. Accessible as [token_name:space_id], [token_name:meeting_uri], [token_name:meeting_code].'),
      '#default_value' => $this->configuration['space_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $space_name = $this->tokenService->replacePlain($this->configuration['space_name']);
    $access_type = $this->configuration['access_type'];
    $space_token_name = $this->tokenService->replacePlain($this->configuration['space_token_name']);


    [$auth_type, $client_id] = $this->validateAndParseAuth('meet operation');
    if (!$auth_type || !$client_id) {
      return;
    }


    $config = ['access_type' => $access_type];

    $space_data = $this->googleMeetService->updateMeetingSpace($auth_type, $client_id, $space_name, $config);

    if (!$space_data) {
      $this->logger->error('Google Meet update space action: Failed to update meeting space @space using @auth_type.', [
        '@space' => $space_name,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    if (!empty($space_token_name)) {
      $this->tokenService->addTokenData($space_token_name, $space_data);
    }
  }

}
