<?php

declare(strict_types=1);

namespace Drupal\eca_google_sheets;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\eca_google\GoogleApiService;
use Google\Service\Sheets;

/**
 * Service for managing Google Sheets API operations.
 */
class GoogleSheetsService {

  use StringTranslationTrait;

  public const string VALUE_INPUT_USER_ENTERED = 'USER_ENTERED';
  public const string VALUE_INPUT_RAW = 'RAW';

  public const string UPDATE_MODE_ENTIRE_ROW = 'entire_row';
  public const string UPDATE_MODE_SPECIFIC_COLUMNS = 'specific_columns';

  /**
   * The Google API service.
   */
  protected GoogleApiService $googleApiService;

  /**
   * The logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a GoogleSheetsService object.
   */
  public function __construct(
    GoogleApiService $google_api_service,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->googleApiService = $google_api_service;
    $this->logger = $logger_factory->get('eca_google_sheets');
  }

  /**
   * Gets a configured Google Sheets service instance.
   *
   * @param string $auth_type
   *   The authentication type ('api_client' or 'service_account').
   * @param string $client_id
   *   The Google API client entity ID.
   *
   * @return \Google\Service\Sheets|null
   *   The Google Sheets service instance or NULL on failure.
   */
  public function getSheetsService(string $auth_type, string $client_id) : ?Sheets {
    $service = $this->googleApiService->getService('sheets', $auth_type, $client_id);
    return $service instanceof Sheets ? $service : NULL;
  }

  /**
   * Validates that the client has Sheets API access.
   *
   * @param string $auth_type
   *   The authentication type ('api_client' or 'service_account').
   * @param string $client_id
   *   The Google API client entity ID.
   *
   * @return bool
   *   TRUE if Sheets API is accessible, FALSE otherwise.
   */
  public function validateApiAccess(string $auth_type, string $client_id): bool {
    return $this->googleApiService->validateApiAccess('sheets', $auth_type, $client_id);
  }

  /**
   * Appends data to a Google Sheet.
   *
   * @param string $auth_type
   *   The authentication type ('api_client' or 'service_account').
   * @param string $client_id
   *   The Google API client entity ID.
   * @param string $spreadsheet_id
   *   The Google Spreadsheet ID.
   * @param string $range
   *   The range to append to (e.g., 'Sheet1' or 'Sheet1!A1:C1').
   * @param array $values
   *   Array of rows, where each row is an array of cell values.
   * @param string $value_input_option
   *   How the input data should be interpreted (RAW or USER_ENTERED).
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function appendToSheet(
    string $auth_type,
    string $client_id,
    string $spreadsheet_id,
    string $range,
    array $values,
    string $value_input_option = GoogleSheetsService::VALUE_INPUT_USER_ENTERED
  ) : bool {
    $sheets_service = $this->getSheetsService($auth_type, $client_id);
    if (!$sheets_service) {
      return FALSE;
    }

    try {
      $value_range = new Sheets\ValueRange();
      $value_range->setValues($values);

      $options = [
        'valueInputOption' => $value_input_option,
      ];

      $sheets_service->spreadsheets_values->append(
        $spreadsheet_id,
        $range,
        $value_range,
        $options
      );

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to append to Google Sheet: @message', ['@message' => $e->getMessage()]);
      return FALSE;
    }
  }

  /**
   * Reads data from a Google Sheet.
   *
   * @param string $auth_type
   *   The authentication type ('api_client' or 'service_account').
   * @param string $client_id
   *   The Google API client entity ID.
   * @param string $spreadsheet_id
   *   The Google Spreadsheet ID.
   * @param string $range
   *   The range to read from (e.g., 'Sheet1!A1:C10').
   *
   * @return array|null
   *   Array of rows with cell values, or NULL on failure.
   */
  public function readFromSheet(
    string $auth_type,
    string $client_id,
    string $spreadsheet_id,
    string $range
  ) : ?array {
    $sheets_service = $this->getSheetsService($auth_type, $client_id);
    if (!$sheets_service) {
      return NULL;
    }

    try {
      $result = $sheets_service->spreadsheets_values->get($spreadsheet_id, $range);
      $values = $result->getValues();

      return $values ?: [];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to read from Google Sheet: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Queries data from a Google Sheet using server-side QUERY function.
   *
   * Creates a temporary hidden sheet within the source spreadsheet, uses QUERY
   * function to filter data server-side, reads results, then deletes the sheet.
   *
   * @param string $auth_type
   *   The authentication type ('api_client' or 'service_account').
   * @param string $client_id
   *   The Google API client entity ID.
   * @param string $source_spreadsheet_id
   *   The source Google Spreadsheet ID to query from.
   * @param string $source_range
   *   The source range (e.g., 'Sheet1!A:Z').
   * @param string $query_expression
   *   The QUERY expression (e.g., 'SELECT * WHERE Col1=34 AND Col3>100').
   *
   * @return array|null
   *   Array of rows with cell values, or NULL on failure.
   */
  public function querySheet(
    string $auth_type,
    string $client_id,
    string $source_spreadsheet_id,
    string $source_range,
    string $query_expression,
    int $sleep_time = 2
  ): ?array {
    $sheets_service = $this->getSheetsService($auth_type, $client_id);
    if (!$sheets_service) {
      return NULL;
    }

    $temp_sheet_id = NULL;

    try {
      // Create temporary sheet inside the source spreadsheet.
      $temp_sheet_title = 'ECA_Query_Temp_' . uniqid();


      // Add a new sheet to the existing spreadsheet.
      $sheet_properties = new Sheets\SheetProperties([
        'title' => $temp_sheet_title,
        'hidden' => TRUE, // Hide the temp sheet from users
      ]);

      $add_sheet_request = new Sheets\AddSheetRequest([
        'properties' => $sheet_properties
      ]);

      $request = new Sheets\Request([
        'addSheet' => $add_sheet_request
      ]);

      $batch_request = new Sheets\BatchUpdateSpreadsheetRequest([
        'requests' => [$request]
      ]);

      $batch_response = $sheets_service->spreadsheets->batchUpdate($source_spreadsheet_id, $batch_request);
      $temp_sheet_id = $batch_response->getReplies()[0]->getAddSheet()->getProperties()->getSheetId();

      // Format the range properly for the QUERY function.
      $formatted_range = $source_range;
      if (!str_contains($source_range, '!')) {
        // If no '!' in range, it's either a sheet name only or a range without sheet.
        // Check if it looks like a range (contains : or cell references) or just a sheet name.
        if (preg_match('/^[A-Za-z0-9_\s]+$/', $source_range) && !preg_match('/[A-Z]+[0-9]/', $source_range)) {
          // Looks like just a sheet name, so add a range to it.
          $formatted_range = "'" . $source_range . "'";
        } else {
          // Looks like a range without sheet name, add the first sheet name.
          $spreadsheet_info = $sheets_service->spreadsheets->get($source_spreadsheet_id);
          $first_sheet = $spreadsheet_info->getSheets()[0];
          $first_sheet_name = $first_sheet->getProperties()->getTitle();
          $formatted_range = "'" . $first_sheet_name . "'!" . $source_range;
        }
      }

      // Smart range detection: Find the actual data boundaries for QUERY.
      $smart_range = $formatted_range;
      try {
        if (preg_match('/^[\'"]?([^\'\"!]+)[\'"]?$/', $formatted_range, $matches)) {
          // This is just a sheet name, we need to detect the data range.
          $sheet_name = $matches[1];

          // First, get basic sheet info to find dimensions.
          $spreadsheet_info = $sheets_service->spreadsheets->get($source_spreadsheet_id);
          $target_sheet = NULL;
          foreach ($spreadsheet_info->getSheets() as $sheet) {
            if ($sheet->getProperties()->getTitle() === $sheet_name) {
              $target_sheet = $sheet;
              break;
            }
          }

          if ($target_sheet) {
            // Get the grid properties to find max rows/columns with data.
            $grid_props = $target_sheet->getProperties()->getGridProperties();
            $max_rows = $grid_props->getRowCount();
            $max_cols = $grid_props->getColumnCount();

            // Convert column number to letter (A, B, C, ..., AA, AB, etc.)
            $last_col = '';
            $col_num = $max_cols - 1; // Zero-indexed
            while ($col_num >= 0) {
              $last_col = chr(65 + ($col_num % 26)) . $last_col;
              $col_num = intval($col_num / 26) - 1;
            }

            // Create a reasonable range that includes all data.
            $smart_range = "'" . $sheet_name . "'!A1:" . $last_col . $max_rows;
          } else {
            $this->logger->warning('Could not find sheet @sheet for smart range detection', ['@sheet' => $sheet_name]);
          }
        }
      }
      catch (\Exception $range_detection_error) {
        $this->logger->warning('Smart range detection failed, using original: @message', [
          '@message' => $range_detection_error->getMessage(),
        ]);
      }

      // Now create the QUERY formula with the smart range and row numbers.
      $query_formula = '=ARRAYFORMULA(QUERY({' . $smart_range . ', ROW(' . $smart_range . ')}, "' . $query_expression . '", 1))';
      $temp_range = "'" . $temp_sheet_title . "'!A1";

      $sheets_service->spreadsheets_values->update(
        $source_spreadsheet_id,
        $temp_range,
        new Sheets\ValueRange(['values' => [[$query_formula]]]),
        ['valueInputOption' => GoogleSheetsService::VALUE_INPUT_USER_ENTERED]
      );

      // Wait for QUERY to process.
      sleep($sleep_time);

      // Read results from temporary sheet.
      // Use the sheet name only to get all data without column/row limits.
      $temp_result_range = "'" . $temp_sheet_title . "'";
      $result = $sheets_service->spreadsheets_values->get($source_spreadsheet_id, $temp_result_range);
      $values = $result->getValues();

      // Fix the last column header to be "_ROW" if we have data.
      if (!empty($values) && !empty($values[0])) {
        $values[0][count($values[0]) - 1] = '_ROW';
      }

      // Clean up: Delete temporary sheet.
      $delete_sheet_request = new Sheets\DeleteSheetRequest([
        'sheetId' => $temp_sheet_id
      ]);

      $delete_request = new Sheets\Request([
        'deleteSheet' => $delete_sheet_request
      ]);

      $delete_batch_request = new Sheets\BatchUpdateSpreadsheetRequest([
        'requests' => [$delete_request]
      ]);

      $sheets_service->spreadsheets->batchUpdate($source_spreadsheet_id, $delete_batch_request);

      return $values ?: [];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to query Google Sheet: @message', ['@message' => $e->getMessage()]);

      // Cleanup: Delete temp sheet if it exists.
      if ($temp_sheet_id && $sheets_service) {
        try {
          $delete_sheet_request = new Sheets\DeleteSheetRequest([
            'sheetId' => $temp_sheet_id
          ]);

          $delete_request = new Sheets\Request([
            'deleteSheet' => $delete_sheet_request
          ]);

          $delete_batch_request = new Sheets\BatchUpdateSpreadsheetRequest([
            'requests' => [$delete_request]
          ]);

          $sheets_service->spreadsheets->batchUpdate($source_spreadsheet_id, $delete_batch_request);
        }
        catch (\Exception $cleanup_error) {
          $this->logger->error('Failed to clean up temporary sheet @id: @message', [
            '@id' => $temp_sheet_id,
            '@message' => $cleanup_error->getMessage()
          ]);
        }
      }

      return NULL;
    }
  }

  /**
   * Updates data in a Google Sheet row.
   *
   * @param string $auth_type
   *   The authentication type ('api_client' or 'service_account').
   * @param string $client_id
   *   The client ID.
   * @param string $spreadsheet_id
   *   The spreadsheet ID.
   * @param string $sheet_range
   *   The sheet name or range.
   * @param int $row_number
   *   The row number to update (1-based).
   * @param array $update_data
   *   The data to update (for entire_row mode: indexed array, for specific_columns mode: associative array with column indices as keys).
   * @param string $update_mode
   *   The update mode ('entire_row' or 'specific_columns').
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function updateSheet(string $auth_type, string $client_id, string $spreadsheet_id, string $sheet_range, int $row_number, array $update_data, string $update_mode): bool {
    try {
      $sheets_service = $this->getSheetsService($auth_type, $client_id);
      if ($sheets_service === NULL) {
        return FALSE;
      }

      if ($update_mode === GoogleSheetsService::UPDATE_MODE_ENTIRE_ROW) {
        // Update entire row
        $range = $sheet_range . '!' . $row_number . ':' . $row_number;

        $value_range = new Sheets\ValueRange([
          'values' => [$update_data]
        ]);

        $sheets_service->spreadsheets_values->update(
          $spreadsheet_id,
          $range,
          $value_range,
          ['valueInputOption' => GoogleSheetsService::VALUE_INPUT_USER_ENTERED]
        );
      } else {
        // Update specific columns
        $updates = [];
        foreach ($update_data as $column_index => $value) {
          $column_letter = $this->getColumnLetter($column_index);
          $cell_range = $sheet_range . '!' . $column_letter . $row_number;

          $updates[] = new Sheets\ValueRange([
            'range' => $cell_range,
            'values' => [[$value]]
          ]);
        }

        if (!empty($updates)) {
          $batch_update_request = new Sheets\BatchUpdateValuesRequest([
            'valueInputOption' => GoogleSheetsService::VALUE_INPUT_USER_ENTERED,
            'data' => $updates
          ]);

          $sheets_service->spreadsheets_values->batchUpdate($spreadsheet_id, $batch_update_request);
        }
      }

      return TRUE;
    } catch (\Exception $e) {
      $this->logger->error('Failed to update sheet: @message', ['@message' => $e->getMessage()]);
      return FALSE;
    }
  }

  /**
   * Creates a new sheet within an existing Google Spreadsheet.
   *
   * @param string $auth_type
   *   The authentication type ('api_client' or 'service_account').
   * @param string $client_id
   *   The client ID.
   * @param string $spreadsheet_id
   *   The Google Spreadsheet ID.
   * @param string $sheet_title
   *   The title for the new sheet.
   *
   * @return array|null
   *   Array containing sheet information on success, NULL on failure.
   */
  public function createSheet(string $auth_type, string $client_id, string $spreadsheet_id, string $sheet_title): ?array {
    try {
      // Get the Sheets service.
      $sheets_service = $this->getSheetsService($auth_type, $client_id);
      if (!$sheets_service) {
        $this->logger->error('Failed to get Sheets service for createSheet operation.');
        return NULL;
      }

      // Create the sheet request.
      $add_sheet_request = new Sheets\AddSheetRequest();
      $sheet_properties = new Sheets\SheetProperties();
      $sheet_properties->setTitle($sheet_title);
      $add_sheet_request->setProperties($sheet_properties);

      // Create the batch update request.
      $batch_update_request = new Sheets\BatchUpdateSpreadsheetRequest();
      $request = new Sheets\Request();
      $request->setAddSheet($add_sheet_request);
      $batch_update_request->setRequests([$request]);

      // Execute the request.
      $response = $sheets_service->spreadsheets->batchUpdate($spreadsheet_id, $batch_update_request);

      if ($response && $response->getReplies() && count($response->getReplies()) > 0) {
        $reply = $response->getReplies()[0];
        $added_sheet = $reply->getAddSheet();

        if ($added_sheet && $added_sheet->getProperties()) {
          $properties = $added_sheet->getProperties();
          return [
            'sheetId' => $properties->getSheetId(),
            'title' => $properties->getTitle(),
            'index' => $properties->getIndex(),
            'sheetType' => $properties->getSheetType(),
            'gridProperties' => [
              'rowCount' => $properties->getGridProperties() ? $properties->getGridProperties()->getRowCount() : 1000,
              'columnCount' => $properties->getGridProperties() ? $properties->getGridProperties()->getColumnCount() : 26,
            ],
          ];
        }
      }

      $this->logger->error('CreateSheet: Unexpected response format from Google Sheets API.');
      return NULL;

    } catch (\Exception $e) {
      $this->logger->error('CreateSheet: Failed to create sheet "@title" in spreadsheet @spreadsheet_id. Error: @error', [
        '@title' => $sheet_title,
        '@spreadsheet_id' => $spreadsheet_id,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Clears values from a Google Sheets range while preserving formatting.
   *
   * @param string $auth_type
   *   The authentication type ('api_client' or 'service_account').
   * @param string $client_id
   *   The client ID.
   * @param string $spreadsheet_id
   *   The Google Spreadsheet ID.
   * @param string $range
   *   The range to clear (e.g., 'Sheet1', 'Sheet1!A1:C10').
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function clearSheet(string $auth_type, string $client_id, string $spreadsheet_id, string $range): bool {
    try {
      // Get the Sheets service.
      $sheets_service = $this->getSheetsService($auth_type, $client_id);
      if (!$sheets_service) {
        $this->logger->error('Failed to get Sheets service for clearSheet operation.');
        return FALSE;
      }

      // Clear values while preserving formatting.
      $clear_values_request = new Sheets\ClearValuesRequest();
      $response = $sheets_service->spreadsheets_values->clear($spreadsheet_id, $range, $clear_values_request);

      if ($response && $response->getClearedRange()) {
        return TRUE;
      } else {
        $this->logger->error('ClearSheet: Failed to clear values from range @range - no response from API', [
          '@range' => $range,
        ]);
        return FALSE;
      }

    } catch (\Exception $e) {
      $this->logger->error('ClearSheet: Exception while clearing values from range @range in spreadsheet @spreadsheet_id. Error: @error', [
        '@range' => $range,
        '@spreadsheet_id' => $spreadsheet_id,
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }


  /**
   * Deletes an entire sheet from a Google Spreadsheet.
   *
   * @param string $auth_type
   *   The authentication type ('api_client' or 'service_account').
   * @param string $client_id
   *   The client ID.
   * @param string $spreadsheet_id
   *   The Google Spreadsheet ID.
   * @param string $sheet_identifier
   *   The sheet title or numeric sheet ID.
   * @param string $identifier_type
   *   Either 'title' or 'id' to indicate how to interpret the identifier.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function deleteSheet(string $auth_type, string $client_id, string $spreadsheet_id, string $sheet_identifier, string $identifier_type = 'title'): bool {
    try {
      // Get the Sheets service.
      $sheets_service = $this->getSheetsService($auth_type, $client_id);
      if (!$sheets_service) {
        $this->logger->error('Failed to get Sheets service for deleteSheet operation.');
        return FALSE;
      }

      // Get the sheet ID we need for deletion.
      $sheet_id = NULL;

      if ($identifier_type === 'id') {
        // User provided numeric sheet ID directly.
        $sheet_id = (int) $sheet_identifier;
      } else {
        // User provided sheet title, we need to look up the sheet ID.
        $sheet_id = $this->getSheetIdByTitle($sheets_service, $spreadsheet_id, $sheet_identifier);
        if ($sheet_id === NULL) {
          $this->logger->error('DeleteSheet: Could not find sheet with title "@title" in spreadsheet @spreadsheet_id.', [
            '@title' => $sheet_identifier,
            '@spreadsheet_id' => $spreadsheet_id,
          ]);
          return FALSE;
        }
      }

      // Create the delete sheet request.
      $delete_sheet_request = new Sheets\DeleteSheetRequest();
      $delete_sheet_request->setSheetId($sheet_id);

      // Create the batch update request.
      $batch_update_request = new Sheets\BatchUpdateSpreadsheetRequest();
      $request = new Sheets\Request();
      $request->setDeleteSheet($delete_sheet_request);
      $batch_update_request->setRequests([$request]);

      // Execute the request.
      $response = $sheets_service->spreadsheets->batchUpdate($spreadsheet_id, $batch_update_request);

      if ($response && $response->getReplies()) {
        return TRUE;
      } else {
        $this->logger->error('DeleteSheet: No response from API when deleting sheet "@identifier".', [
          '@identifier' => $sheet_identifier,
        ]);
        return FALSE;
      }

    } catch (\Exception $e) {
      $this->logger->error('DeleteSheet: Exception while deleting sheet "@identifier" from spreadsheet @spreadsheet_id. Error: @error', [
        '@identifier' => $sheet_identifier,
        '@spreadsheet_id' => $spreadsheet_id,
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Gets a sheet ID by its title.
   *
   * @param \Google\Service\Sheets $sheets_service
   *   The Sheets service instance.
   * @param string $spreadsheet_id
   *   The spreadsheet ID.
   * @param string $sheet_title
   *   The sheet title to look up.
   *
   * @return int|null
   *   The sheet ID or NULL if not found.
   */
  private function getSheetIdByTitle(Sheets $sheets_service, string $spreadsheet_id, string $sheet_title): ?int {
    try {
      // Get spreadsheet metadata to find sheet by title.
      $spreadsheet = $sheets_service->spreadsheets->get($spreadsheet_id);
      $sheets = $spreadsheet->getSheets();

      foreach ($sheets as $sheet) {
        $properties = $sheet->getProperties();
        if ($properties && $properties->getTitle() === $sheet_title) {
          return $properties->getSheetId();
        }
      }

      return NULL;
    } catch (\Exception $e) {
      $this->logger->error('Failed to get sheet ID for title "@title": @error', [
        '@title' => $sheet_title,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Converts a column index to a column letter.
   *
   * @param int $index
   *   The 0-based column index.
   *
   * @return string
   *   The column letter (A, B, C, ..., AA, AB, etc.)
   */
  private function getColumnLetter(int $index): string {
    $letter = '';
    while ($index >= 0) {
      $letter = chr(65 + ($index % 26)) . $letter;
      $index = intval($index / 26) - 1;
    }
    return $letter;
  }

  public function getValueInputOptions() : array {
    return [
      GoogleSheetsService::VALUE_INPUT_USER_ENTERED => $this->t('User Entered (formulas and formatting applied)'),
      GoogleSheetsService::VALUE_INPUT_RAW => $this->t('Raw (exactly as typed, no formatting)'),
    ];
  }



}
