<?php

declare(strict_types=1);

namespace Drupal\eca_google_sheets\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Action\Attribute\Action;
use Drupal\eca\Attribute\EcaAction;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * ECA action plugin for clearing data from Google Sheets ranges.
 */
#[Action(
  id: 'eca_google_sheets_clear_sheet',
  label: new TranslatableMarkup('Google Sheets: Clear Sheet'),
  category: new TranslatableMarkup('Google Sheets'),
)]
#[EcaAction(
  description: new TranslatableMarkup('Clear values from specific ranges within Google Sheets while preserving formatting.')
)]
class ClearSheet extends SheetsActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return $this->getAuthClientIdDefaultConfig()
      + $this->getSpreadsheetIdDefaultConfig()
      + $this->getSheetRangeDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);
    $form = $this->addSpreadsheetIdConfigurationForm($form, $form_state);
    $form = $this->addSheetRangeConfigurationForm($form, $form_state);
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $spreadsheet_id = $this->tokenService->replacePlain($this->configuration['spreadsheet_id']);
    $sheet_range = $this->tokenService->replacePlain($this->configuration['sheet_range']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('sheet operation');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($spreadsheet_id) || empty($sheet_range)) {
      $this->logger->error('ClearSheet: Missing required configuration values.');
      return;
    }


    // Runtime validation of API access.
    if (!$this->googleSheetsService->validateApiAccess($auth_type, $client_id)) {
      $this->logger->error('ClearSheet: Client @client_id lacks required Sheets API access.', ['@client_id' => $client_id]);
      return;
    }

    // Clear the sheet range values.
    $success = $this->googleSheetsService->clearSheet($auth_type, $client_id, $spreadsheet_id, $sheet_range);
    if (!$success) {
      $this->logger->error('ClearSheet: Failed to clear values from range "@range" in spreadsheet @spreadsheet_id.', [
        '@range' => $sheet_range,
        '@spreadsheet_id' => $spreadsheet_id,
      ]);
      return;
    }
  }

}
