<?php

declare(strict_types=1);

namespace Drupal\eca_google_sheets\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Action\Attribute\Action;
use Drupal\eca\Attribute\EcaAction;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * ECA action plugin for creating new sheets within existing Google Spreadsheets.
 */
#[Action(
  id: 'eca_google_sheets_create_sheet',
  label: new TranslatableMarkup('Google Sheets: Create Sheet'),
  category: new TranslatableMarkup('Google Sheets'),
)]
#[EcaAction(
  description:  new TranslatableMarkup('Create a new sheet (tab) within an existing Google Spreadsheet.')
)]
class CreateSheet extends SheetsActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'sheet_title' => '',
      'sheet_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + $this->getSpreadsheetIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);
    $form = $this->addSpreadsheetIdConfigurationForm($form, $form_state);

    $form['sheet_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sheet Title'),
      '#description' => $this->t('The title for the new sheet (tab). Must be unique within the spreadsheet.'),
      '#default_value' => $this->configuration['sheet_title'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['sheet_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sheet Info Token Name'),
      '#description' => $this->t('Optional token name to store information about the created sheet (ID, title, etc.). Leave blank to skip creating this token.'),
      '#default_value' => $this->configuration['sheet_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $spreadsheet_id = $this->tokenService->replacePlain($this->configuration['spreadsheet_id']);
    $sheet_title = $this->tokenService->replacePlain($this->configuration['sheet_title']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('sheet operation');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($spreadsheet_id) || empty($sheet_title)) {
      $this->logger->error('CreateSheet: Missing required configuration values.');
      return;
    }


    // Runtime validation of API access.
    if (!$this->googleSheetsService->validateApiAccess($auth_type, $client_id)) {
      $this->logger->error('CreateSheet: Client @client_id lacks required Sheets API access.', ['@client_id' => $client_id]);
      return;
    }

    // Create the sheet.
    $sheet_info = $this->googleSheetsService->createSheet($auth_type, $client_id, $spreadsheet_id, $sheet_title);
    if ($sheet_info === NULL) {
      $this->logger->error('CreateSheet: Failed to create sheet "@title" in spreadsheet @spreadsheet_id.', [
        '@title' => $sheet_title,
        '@spreadsheet_id' => $spreadsheet_id,
      ]);
      return;
    }

    // Add sheet info token if token name is provided.
    $sheet_token_name = $this->configuration['sheet_token_name'];
    if (!empty($sheet_token_name)) {
      $this->tokenService->addTokenData($sheet_token_name, $sheet_info);
    }
  }

}
