<?php

declare(strict_types=1);

namespace Drupal\eca_google_sheets\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Read data from a Google Sheet.
 */
#[Action(
  id: 'eca_google_sheets_read_sheet',
  label: new TranslatableMarkup('Google Sheets: Read from Sheet'),
  category: new TranslatableMarkup('Google Sheets'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Reads data from a Google Sheet and makes it available as ECA tokens. Specify the sheet range to read specific cells or rows. Supports token replacement for dynamic sheet access.'),
  version_introduced: '1.0.0',
)]
final class ReadFromSheet extends SheetsActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'token_name' => 'sheet_data',
      'headers_token_name' => '',
      'count_token_name' => '',
      'skip_header_row' => FALSE,
      'use_header_as_keys' => FALSE,
    ] + $this->getAuthClientIdDefaultConfig()
      + $this->getSpreadsheetIdDefaultConfig()
      + $this->getSheetRangeDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);
    $form = $this->addSpreadsheetIdConfigurationForm($form, $form_state);
    $form = $this->addSheetRangeConfigurationForm($form, $form_state);

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token Name'),
      '#description' => $this->t('Name for the token that will contain the sheet data. Access data using Google Sheets syntax: [token:Row1:Col1] for first row/column, [token:Row2:Col3] for second row/third column.'),
      '#default_value' => $this->configuration['token_name'],
      '#required' => TRUE,
    ];

    $form['headers_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Headers Token Name'),
      '#description' => $this->t('Optional token name for the header row data. Leave blank to skip creating this token.'),
      '#default_value' => $this->configuration['headers_token_name'],
    ];

    $form['count_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Count Token Name'),
      '#description' => $this->t('Optional token name for the row count. Leave blank to skip creating this token.'),
      '#default_value' => $this->configuration['count_token_name'],
    ];

    $form['skip_header_row'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Skip Header Row'),
      '#description' => $this->t('If checked, the first row will be removed from the returned data (useful when the first row contains column headers).'),
      '#default_value' => $this->configuration['skip_header_row'],
    ];

    $form['use_header_as_keys'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Header Row as Keys'),
      '#description' => $this->t('If checked, the first row will be used as keys for each data row, making it easier to access columns by name (e.g., [sheet_data:Row1:Name] instead of [sheet_data:Row1:Col1]).'),
      '#default_value' => $this->configuration['use_header_as_keys'],
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $spreadsheet_id = $this->tokenService->replacePlain($this->configuration['spreadsheet_id']);
    $sheet_range = $this->tokenService->replacePlain($this->configuration['sheet_range']);
    $token_name = $this->configuration['token_name'];

    // Validate authentication.
    [$auth_type, $client_id] = $this->validateAndParseAuth('read from sheet');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($spreadsheet_id) || empty($sheet_range) || empty($token_name)) {
      $this->logger->error('Google Sheets read action: Missing required configuration (spreadsheet ID, sheet range, or token name).');
      return;
    }


    // Read from Google Sheet.
    $data = $this->googleSheetsService->readFromSheet(
      $auth_type,
      $client_id,
      $spreadsheet_id,
      $sheet_range
    );

    if ($data === NULL) {
      $this->logger->error('Google Sheets read action: Failed to read data from sheet @sheet_id using @auth_type.', [
        '@sheet_id' => $spreadsheet_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Process header row options.
    $skip_header_row = $this->configuration['skip_header_row'];
    $use_header_as_keys = $this->configuration['use_header_as_keys'];
    $processed_data = $data;
    $headers = [];

    if (!empty($data) && ($skip_header_row || $use_header_as_keys)) {
      // Extract the header row if we need it.
      $headers = array_shift($processed_data);

      if ($use_header_as_keys && !empty($headers)) {
        // Convert each data row to use header values as keys.
        $keyed_data = [];
        foreach ($processed_data as $row) {
          $keyed_row = [];
          foreach ($headers as $index => $header) {
            $keyed_row[$header] = $row[$index] ?? '';
          }
          $keyed_data[] = $keyed_row;
        }
        $processed_data = $keyed_data;

        // If we're not skipping the header row, add it back as the first row (but keyed).
        if (!$skip_header_row) {
          $header_as_keyed_row = [];
          foreach ($headers as $header) {
            $header_as_keyed_row[$header] = $header;
          }
          array_unshift($processed_data, $header_as_keyed_row);
        }
      }
      elseif (!$skip_header_row) {
        // We extracted the header but don't want to skip it, so add it back.
        array_unshift($processed_data, $headers);
      }
    } else {
      // No header processing - convert to 1-indexed arrays for consistency with Google Sheets
      $indexed_data = [];
      foreach ($processed_data as $row) {
        if (is_array($row)) {
          $indexed_row = [];
          foreach ($row as $index => $value) {
            $indexed_row['Col' . ($index + 1)] = $value;
          }
          $indexed_data[] = $indexed_row;
        } else {
          $indexed_data[] = $row;
        }
      }
      $processed_data = $indexed_data;
    }

    // Convert to Google Sheets row syntax for full consistency.
    $final_data = [];
    $row_index = 1;
    foreach ($processed_data as $row) {
      $final_data['Row' . $row_index] = $row;
      $row_index++;
    }

    // Make the processed data available as ECA tokens.
    $this->tokenService->addTokenData($token_name, $final_data);

    // Add count token if token name is provided.
    $count_token_name = $this->configuration['count_token_name'];
    if (!empty($count_token_name)) {
      $this->tokenService->addTokenData($count_token_name, count($processed_data));
    }

    // Add headers token if headers were extracted and token name is provided.
    $headers_token_name = $this->configuration['headers_token_name'];
    if (!empty($headers) && !empty($headers_token_name)) {
      $this->tokenService->addTokenData($headers_token_name, $headers);
    }
  }

}
