<?php

declare(strict_types=1);

namespace Drupal\eca_google_sheets\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\DataType\DataTransferObject;
use Drupal\Core\Action\Attribute\Action;
use Drupal\eca\Attribute\EcaAction;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * ECA action plugin for updating Google Sheets data.
 */
#[Action(
  id: 'eca_google_sheets_update_sheet',
  label: new TranslatableMarkup('Google Sheets: Update Sheet'),
  category: new TranslatableMarkup('Google Sheets'),
)]
#[EcaAction(
  description:  new TranslatableMarkup('Update data in Google Sheets by row number with support for entire row or specific column updates.')
)]
class UpdateSheet extends SheetsActionBase {


  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'row_number' => '',
      'update_mode' => 'entire_row',
      'row_data' => '',
      'column_updates' => '',
      'headers_token' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + $this->getSpreadsheetIdDefaultConfig()
      + $this->getSheetRangeDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);
    $form = $this->addSpreadsheetIdConfigurationForm($form, $form_state);
    $form = $this->addSheetRangeConfigurationForm($form, $form_state);

    $form['row_number'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Row Number'),
      '#description' => $this->t('The row number to update (typically from QuerySheet results like [query_data:Row1:_ROW]).'),
      '#default_value' => $this->configuration['row_number'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['update_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Update Mode'),
      '#description' => $this->t('Choose whether to update the entire row or specific columns only.'),
      '#options' => [
        'entire_row' => $this->t('Entire Row'),
        'specific_columns' => $this->t('Specific Columns'),
      ],
      '#default_value' => $this->configuration['update_mode'],
      '#required' => TRUE,
    ];

    $form['row_data'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Row Data Token'),
      '#description' => $this->t('Token containing the complete row data array (for entire row mode). Example: [query_data:Row1]. Any _ROW key will be automatically removed.'),
      '#default_value' => $this->configuration['row_data'],
      '#eca_token_replacement' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="update_mode"]' => ['value' => 'entire_row'],
        ],
        'required' => [
          ':input[name="update_mode"]' => ['value' => 'entire_row'],
        ],
      ],
    ];

    $form['headers_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Headers Token'),
      '#description' => $this->t('Token containing the header row array. Required for header-keyed data in entire row mode, or when using header names in specific columns mode. Example: [query_data_headers]. Alternatively, use the base token name like "query_data" to extract headers from the first row.'),
      '#default_value' => $this->configuration['headers_token'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['column_updates'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Column Updates'),
      '#description' => $this->t('Column updates, one per line. Format: "column:value". Use 1-based column numbers (1, 2, 3...) matching Google Sheets Col1, Col2, Col3 or header names. Example:<br>1:John Smith<br>3:completed<br>Name:Jane Doe'),
      '#default_value' => $this->configuration['column_updates'],
      '#rows' => 4,
      '#eca_token_replacement' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="update_mode"]' => ['value' => 'specific_columns'],
        ],
        'required' => [
          ':input[name="update_mode"]' => ['value' => 'specific_columns'],
        ],
      ],
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $spreadsheet_id = $this->tokenService->replacePlain($this->configuration['spreadsheet_id']);
    $sheet_range = $this->tokenService->replacePlain($this->configuration['sheet_range']);
    $row_number = $this->tokenService->replacePlain($this->configuration['row_number']);
    $update_mode = $this->configuration['update_mode'];

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('sheet operation');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($spreadsheet_id) || empty($sheet_range) || empty($row_number)) {
      $this->logger->error('UpdateSheet: Missing required configuration values.');
      return;
    }


    // Runtime validation of API access.
    if (!$this->googleSheetsService->validateApiAccess($auth_type, $client_id)) {
      $this->logger->error('UpdateSheet: Client @client_id lacks required Sheets API access.', ['@client_id' => $client_id]);
      return;
    }

    // Get headers if available for header-name resolution (used by both modes).
    $headers = [];
    if (!empty($this->configuration['headers_token'])) {
      /** @var DataTransferObject $headers_data */
      $headers_data = $this->tokenService->getTokenData($this->configuration['headers_token']);
      if ($headers_data instanceof DataTransferObject) {
        $headers = $headers_data->toArray();
        if (isset($headers['_ROW'])) {
          unset($headers['_ROW']);
        }
      }
    }

    // Prepare the update data based on mode.
    $update_data = [];

    if ($update_mode === 'entire_row') {
      $row_data_token = $this->tokenService->getTokenData($this->configuration['row_data']);

      if (empty($row_data_token)) {
        $this->logger->error('UpdateSheet: Row data token is required for entire row mode.');
        return;
      }

      // Get the row data array from the token.
      $row_data = $row_data_token->toArray();
      if ($row_data === NULL) {
        $this->logger->error('UpdateSheet: Could not retrieve row data from token @token.', ['@token' => $row_data_token]);
        return;
      }

      // Remove _ROW key if present.
      if (is_array($row_data) && isset($row_data['_ROW'])) {
        unset($row_data['_ROW']);
      }

      // Handle header-keyed arrays.
      if (!empty($headers) && is_array($row_data) && !array_is_list($row_data)) {

        // Convert header-keyed data to indexed array.
        $indexed_data = [];
        foreach ($headers as $index => $header) {
          $indexed_data[$index] = $row_data[$header] ?? '';
        }
        $update_data = array_values($indexed_data);
      } else {
        $update_data = is_array($row_data) ? array_values($row_data) : [$row_data];
      }
    } else {
      // specific_columns mode
      $column_updates = $this->tokenService->replacePlain($this->configuration['column_updates']);

      if (empty($column_updates)) {
        $this->logger->error('UpdateSheet: Column updates are required for specific columns mode.');
        return;
      }

      // Headers are already processed at the beginning of execute()

      // Parse column updates.
      $updates = [];
      $lines = explode("\n", trim($column_updates));
      foreach ($lines as $line) {
        $line = trim($line);
        if (empty($line) || !str_contains($line, ':')) {
          continue;
        }

        [$column, $value] = explode(':', $line, 2);
        $column = trim($column);
        $value = trim($value);

        // Resolve column to numeric index.
        if (is_numeric($column)) {
          $column_index = (int) $column - 1; // Convert from 1-based user input to 0-based API
        } else {
          // Try to find header name in headers array.
          $column_index = array_search($column, $headers, TRUE);
          if ($column_index === FALSE) {
            $this->logger->error('UpdateSheet: Could not resolve column "@column" to numeric index. Skipping this update.', ['@column' => $column]);
            continue;
          }
        }

        $updates[$column_index] = $value;
      }

      if (empty($updates)) {
        $this->logger->error('UpdateSheet: No valid column updates found after processing.');
        return;
      }

      $update_data = $updates;
    }

    // Execute the update.
    $success = $this->googleSheetsService->updateSheet($auth_type, $client_id, $spreadsheet_id, $sheet_range, (int) $row_number, $update_data, $update_mode);

    if (!$success) {
      $this->logger->error('UpdateSheet: Failed to update row @row in sheet @sheet.', [
        '@row' => $row_number,
        '@sheet' => $sheet_range,
      ]);
    }
  }

}
