<?php

declare(strict_types=1);

namespace Drupal\eca_youtube\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_youtube\YouTubeService;

/**
 * Download a specific video caption from YouTube.
 */
#[Action(
  id: 'eca_youtube_get_caption',
  label: new TranslatableMarkup('YouTube: Get Captions'),
  category: new TranslatableMarkup('YouTube'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Downloads a specific video caption from YouTube using a caption ID. Use ListCaptions first to get available caption IDs. Requires OAuth2 authentication and only works for videos you own.'),
  version_introduced: '1.0.0',
)]
final class GetCaption extends YouTubeActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'video_input' => '',
      'caption_id' => '',
      'caption_format' => YouTubeService::TRANSCRIPT_FORMAT_ORIGINAL,
      'caption_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['auth_note'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--warning">' . $this->t('<strong>Important:</strong> This action requires OAuth2 authentication and only works for videos you own or have permission to access. It will not work for other users\' videos.') . '</div>',
    ];

    $form['video_input'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Video ID or URL'),
      '#description' => $this->t('Enter a YouTube video ID or full YouTube URL for a video you own. Supports token replacement.'),
      '#default_value' => $this->configuration['video_input'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['caption_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Captions ID'),
      '#description' => $this->t('The captions track ID to download. Use the List Captions action first to get available IDs, then use a token like <strong>[captions:captions:0:id]</strong> for the first caption or <strong>[captions:captions:1:id]</strong> for the second, etc.'),
      '#default_value' => $this->configuration['caption_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['caption_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Caption Format'),
      '#description' => $this->t('The format to download the caption in.'),
      '#default_value' => $this->configuration['caption_format'],
      '#options' => $this->youTubeService->getTranscriptFormatOptions(),
      '#empty_option' => $this->t('- Original format -'),
    ];

    $form['caption_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Caption Token Name'),
      '#description' => $this->t('Optional token name to store caption data. Accessible as [token_name:content], [token_name:format], [token_name:video_id].'),
      '#default_value' => $this->configuration['caption_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $video_input = $this->tokenService->replacePlain($this->configuration['video_input']);
    $caption_id = $this->tokenService->replacePlain($this->configuration['caption_id']);
    $caption_format = $this->configuration['caption_format'];
    $caption_token_name = $this->tokenService->replacePlain($this->configuration['caption_token_name']);

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('get caption');
    if (!$auth_type || !$client_id) {
      return;
    }

    // Extract video ID from input
    $video_id = $this->validateAndExtractVideoId($video_input, 'get caption');
    if (!$video_id) {
      return;
    }

    if (empty($caption_id)) {
      $this->logger->error('YouTube get caption action: Caption ID is required. Use ListCaptions action first to get available caption IDs.');
      return;
    }


    // Download the caption directly using the caption ID
    $caption_data = $this->youTubeService->downloadTranscript($auth_type, $client_id, $caption_id, $caption_format);

    if (!$caption_data) {
      $this->logger->error('YouTube get caption action: Failed to download caption with ID @caption_id for video @video_id.', [
        '@caption_id' => $caption_id,
        '@video_id' => $video_id,
      ]);
      return;
    }

    // Prepare final caption data
    $final_data = [
      'content' => $caption_data['content'],
      'format' => $caption_data['format'],
      'caption_id' => $caption_id,
      'video_id' => $video_id,
    ];

    // Store caption data in token if requested
    if (!empty($caption_token_name)) {
      $this->tokenService->addTokenData($caption_token_name, $final_data);
    }

  }

}
